<?php
declare(strict_types=1);

/**
 * VisionMedia — Accounting Panel (ENTERPRISE, FINAL)
 * امکانات:
 * - فاکتورها: لیست/مشاهده/دریافت وجه/تسویه/ایجاد جدید
 * - قراردادها: ثبت قرارداد رسمی (طرف اول/دوم) + خروجی PDF/HTML
 * - پروفایل‌ها: مشتری/کارمند (ایجاد/ویرایش/فعال/غیرفعال/حذف/جستجو)
 * - درخواست نقش (onboarding): کارتابل بررسی و تشکیل پرونده
 * - صورتجلسه: ایجاد/ارسال برای امضا (CEO) + خروجی PDF/HTML
 * - دفترکل/گزارش‌ها/جانشینی مدیرعامل
 */

require_once __DIR__ . '/../core/utils.php';
require_once __DIR__ . '/../core/telegram.php';
require_once __DIR__ . '/../core/storage.php';
require_once __DIR__ . '/../config/env.php';

// اگر اینها را قبلاً داری، مانعی ندارد دوبار require شوند:
require_once __DIR__ . '/../modules/accounting.php'; // inv_list(), inv_find(), inv_create(), inv_apply_payment(), inv_render_view_for_customer() ...
require_once __DIR__ . '/../modules/department.php';
require_once __DIR__ . '/../modules/penalties.php';

/* ────────── ثوابت ────────── */
const ACC_PAGE_SIZE       = 10;
const ACC_STATE_FILE      = __DIR__ . '/../storage/acc_state.json';
const CASHBOOK_FILE       = __DIR__ . '/../storage/cashbook.json';
const MINUTES_FILE        = __DIR__ . '/../storage/minutes.json';
const CONTRACTS_FILE      = __DIR__ . '/../storage/contracts.json';
const ROLE_REQUESTS_FILE  = __DIR__ . '/../storage/role_requests.json'; // درخواست‌های نقش (کارمند/مشتری)
const EXPORT_DIR          = __DIR__ . '/../storage/exports';
const LATEFEE_PER_DAY     = 150000; // جریمه تاخیر فاکتور در روز

/* ────────── دسترسی ────────── */
function acc_is_accountant_or_ceo(int $user_id): bool {
    if ($user_id === CEO_UID) return true;
    $emps = storage_load(EMPLOYEES_FILE, []);
    foreach ($emps as $e) {
        if ((int)($e['chat_id']??0) === $user_id) {
            $job = mb_strtolower((string)($e['job']??''));
            if (str_contains($job,'حسابدار') || str_contains($job,'account')) return true;
        }
    }
    return false;
}

/* ────────── State ────────── */
function acc_state_load(): array {
    if (!is_file(ACC_STATE_FILE)) return [];
    $j = json_decode((string)@file_get_contents(ACC_STATE_FILE), true);
    return is_array($j)?$j:[];
}
function acc_state_save(array $s): bool { return (bool)@file_put_contents(ACC_STATE_FILE, json_encode($s, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT)); }
function acc_state_set(int $uid, string $key, $val): void { $s=acc_state_load(); $s[$uid][$key]=$val; acc_state_save($s); }
function acc_state_get(int $uid, string $key, $def=null){ $s=acc_state_load(); return $s[$uid][$key] ?? $def; }
function acc_state_clear(int $uid): void { $s=acc_state_load(); unset($s[$uid]); acc_state_save($s); }

/* ────────── فایل‌های دیتایی کمکی ────────── */
function acc_json_load(string $path): array { return is_file($path)? (json_decode((string)@file_get_contents($path), true) ?: []) : []; }
function acc_json_save(string $path, array $data): bool { return (bool)@file_put_contents($path, json_encode($data, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT)); }

/* ────────── UI Helpers ────────── */
function acc_back_kb(string $to): array { return kb_inline([[ kb_btn('◀️ بازگشت', $to) ]]); }
function acc_paginate_kb(string $base, int $page, int $pages): array {
    if ($pages<=1) return kb_inline([]);
    $row=[]; if ($page>1) $row[] = kb_btn('◀️ قبلی', "{$base}:p:".($page-1)); if ($page<$pages) $row[] = kb_btn('بعدی ▶️', "{$base}:p:".($page+1));
    return kb_inline([ $row ]);
}
function acc_person_title_by_id(int $id, string $type): string {
    if ($type==='employee') {
        $all = storage_load(EMPLOYEES_FILE, []);
        foreach ($all as $e) if ((int)($e['chat_id']??0)===$id) return (string)($e['full_name']??('کارمند '.to_persian_num($id)));
        return 'کارمند '.to_persian_num($id);
    }
    // customer
    $all = storage_load(CUSTOMERS_FILE, []);
    foreach ($all as $c) if ((int)($c['chat_id']??0)===$id) return (string)($c['brand']??('مشتری '.to_persian_num($id)));
    return 'مشتری '.to_persian_num($id);
}

/* ────────── منوی اصلی پنل حسابداری ────────── */
if (!function_exists('accountant_menu_main')) {
    function accountant_menu_main(int $chat_id): void {
        $kb = kb_inline([
            [ kb_btn('🧾 فاکتورها', 'acc:invoices'), kb_btn('📑 قراردادها', 'acc:contracts') ],
            [ kb_btn('👥 مشتریان', 'acc:customers'), kb_btn('👤 پرسنل', 'acc:employees') ],
            [ kb_btn('📝 صورتجلسه', 'acc:minutes'), kb_btn('💼 دفترکل', 'acc:cashbook') ],
            [ kb_btn('📬 درخواست‌های نقش', 'acc:roles'), kb_btn('📊 گزارش‌ها', 'acc:reports') ],
            [ kb_btn('📈 خلاصه مدیریتی (جانشین CEO)', 'acc:exec') ],
        ]);
        $msg = "💼 <b>پنل حسابداری (سازمانی)</b>\n"
             . "از اینجا همهٔ عملیات مالی، قرارداد، پرونده‌سازی و گزارش‌ها دستت هست ✨";
        tg_send_kb_html($chat_id, $msg, $kb);
    }
}

/* ────────── ناوبری اولیه کال‌بک‌ها ────────── */
if (!function_exists('accountant_handle_callback')) {
    function accountant_handle_callback(int $chat_id, int $user_id, string $data): bool {
        if (!acc_is_accountant_or_ceo($user_id)) return false;
        if (!str_starts($data,'acc:')) return false;

        // سطح اول
        if ($data==='acc:back')       { accountant_menu_main($chat_id); return true; }
        if ($data==='acc:invoices')   { acc_invoices_list($chat_id,1,null); return true; }
        if ($data==='acc:contracts')  { acc_contracts_list($chat_id,1,null); return true; }
        if ($data==='acc:customers')  { acc_customers_menu($chat_id); return true; }
        if ($data==='acc:employees')  { acc_employees_menu($chat_id); return true; }
        if ($data==='acc:minutes')    { acc_minutes_menu($chat_id); return true; }
        if ($data==='acc:cashbook')   { acc_cashbook_menu($chat_id); return true; }
        if ($data==='acc:roles')      { acc_roles_inbox($chat_id,1); return true; }
        if ($data==='acc:reports')    { acc_reports_menu($chat_id); return true; }
        if ($data==='acc:exec')       { acc_exec_summary($chat_id); return true; }

        // ادامهٔ مسیریابی در بخش‌های بعدی این فایل پیاده‌سازی شده (بخش 2/3 و 3/3)
        return false;
    }
}

/* ────────── ورودی متنی (فرم‌های مرحله‌ای) ────────── */
if (!function_exists('accountant_handle_text')) {
    function accountant_handle_text(int $chat_id, int $user_id, string $text): bool {
        if (!acc_is_accountant_or_ceo($user_id)) return false;

        // مسیریاب فرم‌های مرحله‌ای — در بخش‌های 2/3 و 3/3 تکمیل می‌شود
        // اولویت: دریافت وجه فاکتور
        $recv = acc_state_get($user_id,'receive');
        if ($recv && ($recv['step']??'')==='await_amount') { acc_invoice_receive_apply($chat_id,$user_id,$text); return true; }

        // ساخت فاکتور
        $inv = acc_state_get($user_id,'inv');
        if ($inv) { acc_invoice_create_collect($chat_id,$user_id,$text); return true; }

        // ساخت قرارداد
        $con = acc_state_get($user_id,'contract');
        if ($con) { acc_contract_create_collect($chat_id,$user_id,$text); return true; }

        // ویرایش پروفایل‌ها
        $pf = acc_state_get($user_id,'profile');
        if ($pf) { acc_profile_collect($chat_id,$user_id,$text); return true; }

        // صورتجلسه
        $min = acc_state_get($user_id,'min');
        if ($min) { acc_minutes_collect($chat_id,$user_id,$text); return true; }

        // دفترکل
        $cb = acc_state_get($user_id,'cb');
        if ($cb) { acc_cb_add_collect($chat_id,$user_id,$text); return true; }

        // جست‌وجوی نام (سریع)
        if (mb_stripos($text,'جستجو ')===0) {
            $q = trim(mb_substr($text, 7));
            if ($q!=='') { acc_quick_search($chat_id, $q); return true; }
        }

        // پیش‌فرض: منوی حسابداری
        accountant_menu_main($chat_id);
        return true;
    }
}

/* ────────── منوهای ثانویه ساده (placeholder؛ در بخش‌های بعد پر می‌شوند) ────────── */
function acc_customers_menu(int $chat_id): void {
    $kb = kb_inline([
        [ kb_btn('📋 فهرست مشتریان', 'acc:c:list'), kb_btn('➕ مشتری جدید', 'acc:c:new') ],
        [ kb_btn('🔎 جستجو نام', 'acc:c:search') ],
        [ kb_btn('◀️ بازگشت', 'acc:back') ],
    ]);
    tg_send_kb_html($chat_id, "👥 <b>مشتریان</b>", $kb);
}
function acc_employees_menu(int $chat_id): void {
    $kb = kb_inline([
        [ kb_btn('📋 فهرست پرسنل', 'acc:e:list'), kb_btn('➕ کارمند جدید', 'acc:e:new') ],
        [ kb_btn('🔎 جستجو نام', 'acc:e:search') ],
        [ kb_btn('◀️ بازگشت', 'acc:back') ],
    ]);
    tg_send_kb_html($chat_id, "👤 <b>پرسنل</b>", $kb);
}
function acc_minutes_menu(int $chat_id): void {
    $kb = kb_inline([
        [ kb_btn('🆕 صورتجلسه جدید', 'acc:min:new') ],
        [ kb_btn('📋 همه', 'acc:min:list:all'), kb_btn('🟡 منتظر امضا', 'acc:min:list:pending') ],
        [ kb_btn('🟢 ابلاغ‌شده', 'acc:min:list:approved'), kb_btn('🔴 ردشده', 'acc:min:list:rejected') ],
        [ kb_btn('◀️ بازگشت', 'acc:back') ],
    ]);
    tg_send_kb_html($chat_id, "📝 <b>صورتجلسه</b>", $kb);
}
function acc_cashbook_menu(int $chat_id): void {
    $kb = kb_inline([
        [ kb_btn('➕ ثبت درآمد', 'acc:cb:add:income'), kb_btn('➕ ثبت هزینه', 'acc:cb:add:expense') ],
        [ kb_btn('📋 لیست همه', 'acc:cb:list:all'), kb_btn('💰 فقط درآمد', 'acc:cb:list:income') ],
        [ kb_btn('💸 فقط هزینه', 'acc:cb:list:expense') ],
        [ kb_btn('◀️ بازگشت', 'acc:back') ],
    ]);
    tg_send_kb_html($chat_id, "💼 <b>دفترکل</b>", $kb);
}

/* ────────── خلاصه مدیریتی (جانشین CEO) ────────── */
function acc_exec_summary(int $chat_id): void {
    $emps = storage_load(EMPLOYEES_FILE, []);
    $cust = storage_load(CUSTOMERS_FILE, []);
    $invs = inv_list();
    $staff_active = count(array_filter($emps, fn($e)=> ($e['active']??true)));
    $customers_active = count($cust);
    $open_invs = array_filter($invs, fn($i)=> in_array(($i['status']??'unpaid'), ['unpaid','partial'], true));
    $open_count = count($open_invs);
    $remain_total=0; foreach ($open_invs as $inv){ $remain_total += (int)(inv_remain_info((int)$inv['id'])['remain']??0); }

    $msg = "📈 <b>خلاصه مدیریتی</b>\n"
         . "👥 پرسنل فعال: <b>".to_persian_num($staff_active)."</b>\n"
         . "🤝 مشتریان: <b>".to_persian_num($customers_active)."</b>\n"
         . "🧾 فاکتورهای باز: <b>".to_persian_num($open_count)."</b>\n"
         . "💰 مانده کل: <b>".money($remain_total)."</b>";
    tg_send_kb_html($chat_id,$msg, acc_back_kb('acc:back'));
}

/* ────────── جستجوی سریع نام (کارمند/مشتری) ────────── */
function acc_quick_search(int $chat_id, string $q): void {
    $qn = mb_strtolower(trim($q));

    $emps = storage_load(EMPLOYEES_FILE, []);
    $cust = storage_load(CUSTOMERS_FILE, []);

    $em = array_values(array_filter($emps, fn($e)=> mb_stripos((string)($e['full_name']??''), $q)!==false));
    $cu = array_values(array_filter($cust, fn($c)=> mb_stripos((string)($c['brand']??''), $q)!==false));

    $rows=[];
    foreach ($em as $e) {
        $name = $e['full_name'] ?: '—';
        $rows[] = [ kb_btn("👤 {$name}", "acc:e:show:".(int)($e['chat_id']??0)) ];
    }
    foreach ($cu as $c) {
        $brand = $c['brand'] ?: '—';
        $rows[] = [ kb_btn("🏢 {$brand}", "acc:c:show:".(int)($c['chat_id']??0)) ];
    }
    if (!$rows) { tg_send_kb_html($chat_id, "نتیجه‌ای برای «<b>".tg_escape($q)."</b>» پیدا نشد.", acc_back_kb('acc:back')); return; }
    $rows[]=[ kb_btn('◀️ بازگشت','acc:back') ];
    tg_send_kb_html($chat_id, "🔎 نتایج برای «<b>".tg_escape($q)."</b>»", ['inline_keyboard'=>$rows]);
}
/* ===================== فاکتورها ===================== */

function acc_invoices_list(int $chat_id, int $page=1, ?int $cust_id=null): void {
    $invoices = inv_list();
    if ($cust_id) $invoices = array_values(array_filter($invoices, fn($i)=>(int)($i['customer_chat_id']??0)===$cust_id));
    if (!$invoices) { tg_send_kb_html($chat_id, $cust_id? 'هیچ فاکتوری برای این مشتری نیست.' : 'هنوز فاکتوری ثبت نشده.', acc_back_kb($cust_id? "acc:c:show:{$cust_id}" : 'acc:back')); return; }

    usort($invoices, fn($a,$b)=>($b['created_ts']??0) <=> ($a['created_ts']??0));
    $total=count($invoices); $pages=(int)ceil($total/ACC_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($invoices, ($page-1)*ACC_PAGE_SIZE, ACC_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $inv) {
        $id=(int)$inv['id'];
        $custName = acc_person_title_by_id((int)($inv['customer_chat_id']??0), 'customer');
        $st = $inv['status'] ?? 'unpaid';
        $remain = inv_remain_info($id);
        $rows[]=[ kb_btn("🧾 #".to_persian_num($id)." — {$custName} — {$st} — مانده: ".money((int)$remain['remain']), "acc:inv:view:{$id}") ];
    }
    $nav=acc_paginate_kb($cust_id? "acc:inv:for:{$cust_id}" : 'acc:inv', $page,$pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows,$nav);
    $rows[]=[ kb_btn('➕ فاکتور جدید', 'acc:inv:new') ];
    $rows[]=[ kb_btn('◀️ بازگشت', $cust_id? "acc:c:show:{$cust_id}" : 'acc:back') ];
    tg_send_kb_html($chat_id, "🧾 <b>فهرست فاکتورها</b>", ['inline_keyboard'=>$rows]);
}

function acc_invoice_view(int $chat_id, int $inv_id): void {
    $inv = inv_find($inv_id);
    if (!$inv) { tg_send_html($chat_id,'⛔️ فاکتور یافت نشد.'); return; }

    $custName = acc_person_title_by_id((int)($inv['customer_chat_id']??0), 'customer');
    $view = "🧾 <b>جزئیات فاکتور #".to_persian_num($inv_id)."</b>\n"
          . "مشتری: <b>{$custName}</b>\n"
          . inv_render_view_for_customer($inv_id);

    $kb = kb_inline([
        [ kb_btn('➕ دریافت وجه', "acc:inv:recv:{$inv_id}"), kb_btn('✔️ تسویه کل', "acc:inv:settle:{$inv_id}") ],
        [ kb_btn('⏱ اعمال جریمه تأخیر تا امروز', "acc:inv:latefee:{$inv_id}") ],
        [ kb_btn('◀️ بازگشت', 'acc:invoices') ],
    ]);
    tg_send_kb_html($chat_id, $view, $kb);
}

function acc_invoice_receive_prompt(int $chat_id, int $uid, int $inv_id): void {
    acc_state_set($uid, 'receive', ['inv'=>$inv_id, 'step'=>'await_amount']);
    tg_send_kb_html($chat_id, "مبلغ دریافتی را به تومان وارد کن (فقط عدد) 💵", acc_back_kb("acc:inv:view:{$inv_id}"));
}
function acc_invoice_receive_apply(int $chat_id, int $uid, string $text): void {
    $st = acc_state_get($uid,'receive');
    if (!$st || ($st['step']??'')!=='await_amount') { tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $inv_id = (int)$st['inv'];
    $amount = (int)preg_replace('~\D+~','', $text);
    if ($amount<=0) { tg_send_html($chat_id,'⛔️ مبلغ نامعتبر. دوباره بفرست.'); return; }
    $ok = inv_apply_payment($inv_id, $amount);
    acc_state_clear($uid);
    tg_send_html($chat_id, $ok? "✅ دریافت وجه ثبت شد." : "⛔️ ثبت نشد.");
    acc_invoice_view($chat_id,$inv_id);
}
function acc_invoice_settle_all(int $chat_id, int $inv_id): void {
    $r = inv_remain_info($inv_id);
    $ok = inv_apply_payment($inv_id, (int)$r['remain']);
    tg_send_html($chat_id, $ok? "✅ فاکتور تسویه شد." : "⛔️ انجام نشد.");
    acc_invoice_view($chat_id,$inv_id);
}
function acc_invoice_apply_latefee(int $chat_id, int $inv_id): void {
    tg_send_html($chat_id, "ℹ️ جریمه تأخیر روزانه به‌صورت خودکار توسط کران اعمال می‌شود (+ ".money(LATEFEE_PER_DAY)." در هر روز).");
    acc_invoice_view($chat_id,$inv_id);
}

/* ایجاد فاکتور جدید — مرحله‌ای */
function acc_invoice_new_start(int $chat_id, int $uid): void {
    $cust = storage_load(CUSTOMERS_FILE, []);
    if (!$cust) { tg_send_kb_html($chat_id,'⛔️ هنوز مشتری ثبت نشده.', acc_back_kb('acc:invoices')); return; }
    usort($cust, fn($a,$b)=>strcmp(($a['brand']??''), ($b['brand']??'')));
    $rows=[]; foreach ($cust as $c) {
        $cid=(int)($c['chat_id']??0); $brand=$c['brand'] ?: '—';
        $rows[]=[ kb_btn("🏢 {$brand}", "acc:inv:new:c:{$cid}") ];
    }
    $rows[]=[ kb_btn('◀️ بازگشت','acc:invoices') ];
    acc_state_set($uid, 'inv', ['step'=>'pick_customer']);
    tg_send_kb_html($chat_id,"برای چه مشتری؟", ['inline_keyboard'=>$rows]);
}
function acc_invoice_create_collect(int $chat_id, int $uid, string $text): void {
    $st = acc_state_get($uid,'inv'); if (!$st) return;
    $step = $st['step'] ?? '';
    if ($step==='await_title') {
        $st['title']=trim($text); $st['step']='await_amount';
        acc_state_set($uid,'inv',$st);
        tg_send_html($chat_id,'مبلغ فاکتور (تومان — فقط عدد) را وارد کن:');
        return;
    }
    if ($step==='await_amount') {
        $num = (int)preg_replace('~\D+~','', $text);
        if ($num<=0) { tg_send_html($chat_id,'⛔️ مبلغ نامعتبر.'); return; }
        $st['amount']=$num; $st['step']='confirm';
        acc_state_set($uid,'inv',$st);
        $custName = acc_person_title_by_id((int)$st['customer'], 'customer');
        $kb = kb_inline([[ kb_btn('✅ ثبت فاکتور', 'acc:inv:new:save'), kb_btn('❌ لغو', 'acc:inv:new:cancel') ]]);
        $msg="🧾 پیش‌نویس فاکتور:\nمشتری: <b>{$custName}</b>\nعنوان: <b>".tg_escape($st['title'])."</b>\nمبلغ: <b>".money($num)."</b>";
        tg_send_kb_html($chat_id,$msg,$kb);
        return;
    }
}
function acc_invoice_create_save(int $chat_id, int $uid): void {
    $st = acc_state_get($uid,'inv'); if (!$st || ($st['step']??'')!=='confirm') { tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $inv_id = inv_create((int)$st['customer'], $st['title'], (int)$st['amount']);
    acc_state_clear($uid);
    tg_send_html($chat_id, $inv_id? "✅ فاکتور #".to_persian_num($inv_id)." ساخته شد." : "⛔️ فاکتور ساخته نشد.");
    acc_invoices_list($chat_id,1,null);
}

/* ===================== قراردادها ===================== */

function acc_contracts_list(int $chat_id, int $page=1, ?int $cust_id=null): void {
    $arr = acc_json_load(CONTRACTS_FILE);
    if ($cust_id) $arr = array_values(array_filter($arr, fn($x)=>(int)($x['customer']??0)===$cust_id));
    if (!$arr) { tg_send_kb_html($chat_id, $cust_id? 'قراردادی برای این مشتری نیست.' : 'هنوز قراردادی ثبت نشده.', acc_back_kb($cust_id? "acc:c:show:{$cust_id}" : 'acc:back')); return; }
    usort($arr, fn($a,$b)=>($b['ts']??0) <=> ($a['ts']??0));
    $total=count($arr); $pages=(int)ceil($total/ACC_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($arr, ($page-1)*ACC_PAGE_SIZE, ACC_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $c) {
        $id=(int)$c['id']; $title=$c['title']??'—';
        $custName = acc_person_title_by_id((int)($c['customer']??0), 'customer');
        $rows[]=[ kb_btn("📑 #".to_persian_num($id)." — {$custName} — ".tg_escape($title), "acc:con:view:{$id}") ];
    }
    $nav=acc_paginate_kb($cust_id? "acc:con:for:{$cust_id}" : 'acc:con', $page,$pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows,$nav);
    $rows[]=[ kb_btn('➕ قرارداد جدید', 'acc:con:new') ];
    $rows[]=[ kb_btn('◀️ بازگشت', $cust_id? "acc:c:show:{$cust_id}" : 'acc:back') ];
    tg_send_kb_html($chat_id, "📑 <b>فهرست قراردادها</b>", ['inline_keyboard'=>$rows]);
}

function acc_contract_view(int $chat_id, int $id): void {
    $arr = acc_json_load(CONTRACTS_FILE);
    $c=null; foreach ($arr as $x) if ((int)($x['id']??0)===$id) { $c=$x; break; }
    if (!$c) { tg_send_html($chat_id,'⛔️ قرارداد یافت نشد.'); return; }

    // طرف اول/دوم رسمی
    $company = [
        'name' => 'البرز مدیا دوراندیش (ویژن مدیا)',
        'reg'  => '۴۲۵۶۳',
        'nid'  => '۱۴۰۱۱۱۲۸۲۲۲',
        'addr' => 'کرج ، جهانشهر، میدان سپاه ، مجتمع تجاری گلستان',
        'phone'=> '۰۹۳۳۱۹۰۳۸۷۷ / ۰۹۹۸۱۳۸۱۷۳ / ۰۲۶۳۴۴۲۵۸۸ / ۰۲۱۹۱۳۰۵۸۸۳',
        'site' => 'www.visionmedia.ir'
    ];
    $custName = acc_person_title_by_id((int)($c['customer']??0), 'customer');

    $body = "📑 <b>قرارداد #".to_persian_num((int)$c['id'])."</b>\n"
          . "عنوان: <b>".tg_escape((string)($c['title']??'')) ."</b>\n"
          . "تاریخ: <b>".jdate('Y/m/d',$c['ts']??time())."</b>\n\n"
          . "🔷 <b>طرف اول (شرکت)</b>\n"
          . "{$company['name']} (شماره ثبت {$company['reg']} / شناسه ملی {$company['nid']})\n"
          . "{$company['addr']}\n"
          . "تلفن: {$company['phone']} — وبسایت: {$company['site']}\n\n"
          . "🔶 <b>طرف دوم (خریدار/مشتری)</b>\n"
          . "<b>{$custName}</b>\n\n"
          . "📝 مفاد: \n".tg_escape((string)($c['terms']??'—'))."\n\n"
          . "✍️ نمایندگان مجاز امضا: مدیرعامل / حسابدار سازمان\n"
          . "📎 پیوست‌ها: ".(count($c['attachments']??[])?:0)." فایل";

    $kb = kb_inline([
        [ kb_btn('🖨 خروجی PDF/HTML', "acc:con:export:{$id}") ],
        [ kb_btn('◀️ بازگشت','acc:contracts') ],
    ]);
    tg_send_kb_html($chat_id, $body, $kb);
}

/* ایجاد قرارداد جدید — مرحله‌ای */
function acc_contract_new_start(int $chat_id, int $uid): void {
    $cust = storage_load(CUSTOMERS_FILE, []);
    if (!$cust) { tg_send_kb_html($chat_id,'⛔️ مشتری ثبت نشده.', acc_back_kb('acc:contracts')); return; }
    usort($cust, fn($a,$b)=>strcmp(($a['brand']??''), ($b['brand']??'')));
    $rows=[]; foreach ($cust as $c) {
        $cid=(int)($c['chat_id']??0); $brand=$c['brand'] ?: '—';
        $rows[]=[ kb_btn("🏢 {$brand}", "acc:con:new:c:{$cid}") ];
    }
    $rows[]=[ kb_btn('◀️ بازگشت','acc:contracts') ];
    acc_state_set($uid, 'contract', ['step'=>'pick_customer']);
    tg_send_kb_html($chat_id, "قرارداد برای کدام مشتری؟", ['inline_keyboard'=>$rows]);
}
function acc_contract_create_collect(int $chat_id, int $uid, string $text): void {
    $st = acc_state_get($uid,'contract'); if (!$st) return;
    $step = $st['step'] ?? '';
    if ($step==='await_title') {
        $st['title']=trim($text); $st['step']='await_terms';
        acc_state_set($uid,'contract',$st);
        tg_send_html($chat_id, "📝 مفاد قرارداد را کامل و رسمی بنویس (مواد، بندها، تعهدات، وجه التزام، فسخ، محرمانگی...).");
        return;
    }
    if ($step==='await_terms') {
        $st['terms']=trim($text); $st['step']='confirm';
        acc_state_set($uid,'contract',$st);
        $custName = acc_person_title_by_id((int)$st['customer'], 'customer');
        $kb = kb_inline([[ kb_btn('✅ ثبت قرارداد', 'acc:con:new:save'), kb_btn('❌ لغو', 'acc:con:new:cancel') ]]);
        $msg="📑 پیش‌نویس قرارداد:\nمشتری: <b>{$custName}</b>\nعنوان: <b>".tg_escape($st['title'])."</b>\n— مفاد وارد شد.";
        tg_send_kb_html($chat_id,$msg,$kb);
        return;
    }
}
function acc_contract_create_save(int $chat_id, int $uid): void {
    $st = acc_state_get($uid,'contract'); if (!$st || ($st['step']??'')!=='confirm') { tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $arr = acc_json_load(CONTRACTS_FILE);
    $id = (int)(time().mt_rand(100,999));
    $arr[] = [
        'id'=>$id,
        'customer'=>(int)$st['customer'],
        'title'=>$st['title'],
        'terms'=>$st['terms'],
        'attachments'=>[],
        'ts'=>time(),
        'by'=>$uid
    ];
    acc_json_save(CONTRACTS_FILE,$arr);
    acc_state_clear($uid);
    tg_send_html($chat_id, "✅ قرارداد #".to_persian_num($id)." ثبت شد.");
    acc_contracts_list($chat_id,1,null);
}

/* خروجی قرارداد به PDF/HTML */
function acc_contract_export(int $chat_id, int $id): void {
    $arr = acc_json_load(CONTRACTS_FILE);
    $c=null; foreach ($arr as $x) if ((int)($x['id']??0)===$id) { $c=$x; break; }
    if (!$c) { tg_send_html($chat_id,'⛔️ قرارداد یافت نشد.'); return; }

    $companyName = "البرز مدیا دوراندیش (ویژن مدیا)";
    $custName = acc_person_title_by_id((int)($c['customer']??0), 'customer');
    $html = "<html><head><meta charset='utf-8'><style>
        body{font-family:Tahoma, sans-serif;direction:rtl;line-height:1.9}
        h1{font-size:18px;margin:0 0 10px}
        .sec{margin:10px 0;padding:10px;border:1px solid #ddd;border-radius:8px}
        .muted{color:#666}
    </style></head><body>
    <h1>قرارداد شماره ".to_persian_num((string)$c['id'])."</h1>
    <div class='sec'><b>عنوان:</b> ".tg_escape((string)$c['title'])."</div>
    <div class='sec'><b>طرف اول (شرکت):</b> {$companyName}</div>
    <div class='sec'><b>طرف دوم (خریدار/مشتری):</b> {$custName}</div>
    <div class='sec'><b>مفاد قرارداد:</b><br>".nl2br(tg_escape((string)$c['terms']))."</div>
    <div class='sec muted'>تاریخ: ".jdate('Y/m/d H:i', (int)($c['ts']??time()))." | امضا: مدیرعامل / حسابدار</div>
    </body></html>";

    // اگر Dompdf نصب است، PDF بساز
    if (class_exists('Dompdf\Dompdf')) {
        if (!is_dir(EXPORT_DIR)) @mkdir(EXPORT_DIR, 0775, true);
        $file = EXPORT_DIR . "/contract_{$id}.pdf";
        $dompdf = new Dompdf\Dompdf(); $dompdf->loadHtml($html, 'UTF-8'); $dompdf->setPaper('A4'); $dompdf->render();
        @file_put_contents($file, $dompdf->output());
        tg_send_html($chat_id, "✅ خروجی PDF آماده شد.\n<code>{$file}</code>");
    } else {
        // HTML قابل‌پرینت
        tg_send_html($chat_id, "ℹ️ ماژول PDF در هاست فعال نیست؛ نسخهٔ HTML قابل‌پرینت آماده است.\n".$html);
    }
}

/* ===================== مشتریان (CRUD + جستجو نام) ===================== */

function acc_customers_list(int $chat_id, int $page=1): void {
    $all = storage_load(CUSTOMERS_FILE, []);
    if (!$all) { tg_send_kb_html($chat_id,'هنوز مشتری ثبت نشده.', acc_back_kb('acc:customers')); return; }
    usort($all, fn($a,$b)=>strcmp(($a['brand']??''), ($b['brand']??'')));
    $total=count($all); $pages=(int)ceil($total/ACC_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($all, ($page-1)*ACC_PAGE_SIZE, ACC_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $c) {
        $cid=(int)($c['chat_id']??0);
        $brand=$c['brand'] ?: '—'; $ig=$c['instagram'] ?: '—';
        $rows[]=[ kb_btn("🏢 {$brand} — {$ig}", "acc:c:show:{$cid}") ];
    }
    $nav=acc_paginate_kb('acc:c:list',$page,$pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows,$nav);
    $rows[]=[ kb_btn('◀️ بازگشت','acc:customers') ];
    tg_send_kb_html($chat_id,"📋 <b>فهرست مشتریان</b>", ['inline_keyboard'=>$rows]);
}
function acc_customer_profile(int $chat_id, int $cust_id): void {
    $all=storage_load(CUSTOMERS_FILE,[]); $c=null; foreach($all as $x) if((int)($x['chat_id']??0)===$cust_id){$c=$x;break;}
    if(!$c){ tg_send_html($chat_id,'⛔️ مشتری یافت نشد.'); return; }
    $brand=$c['brand']?:'—'; $ig=$c['instagram']?:'—'; $active=($c['active']??true)?'فعال 🟢':'غیرفعال 🔴';
    $kb = kb_inline([
        [ kb_btn('✏️ ویرایش', "acc:c:edit:{$cust_id}"), kb_btn(($c['active']??true)?'🔒 غیرفعال':'🔓 فعال', "acc:c:toggle:{$cust_id}") ],
        [ kb_btn('🧾 فاکتورها', "acc:inv:for:{$cust_id}"), kb_btn('📑 قراردادها', "acc:con:for:{$cust_id}") ],
        [ kb_btn('🗑 حذف', "acc:c:del:{$cust_id}") ],
        [ kb_btn('◀️ بازگشت','acc:c:list') ],
    ]);
    $msg="🏢 <b>{$brand}</b>\nاینستاگرام: <code>{$ig}</code>\nوضعیت: {$active}";
    tg_send_kb_html($chat_id,$msg,$kb);
}
function acc_customer_new_start(int $chat_id, int $uid): void {
    acc_state_set($uid,'profile',['type'=>'customer','step'=>'await_brand']);
    tg_send_kb_html($chat_id,'نام برند/مشتری را وارد کن:', acc_back_kb('acc:customers'));
}
function acc_profile_collect(int $chat_id, int $uid, string $text): void {
    $st = acc_state_get($uid,'profile'); if(!$st) return;
    if (($st['type']??'')==='customer') {
        $step=$st['step']??'';
        if ($step==='await_brand') {
            $st['brand']=trim($text); $st['step']='await_instagram'; acc_state_set($uid,'profile',$st);
            tg_send_html($chat_id,'آی‌دی اینستاگرام (بدون @) را وارد کن (یا خط تیره -):');
            return;
        }
        if ($step==='await_instagram') {
            $st['instagram']=trim($text); $st['step']='confirm'; acc_state_set($uid,'profile',$st);
            $kb=kb_inline([[ kb_btn('✅ ثبت', 'acc:c:new:save'), kb_btn('❌ لغو', 'acc:profile:cancel') ]]);
            $msg="پیش‌نویس مشتری:\nنام: <b>".tg_escape($st['brand'])."</b>\nاینستاگرام: <b>".tg_escape($st['instagram'])."</b>";
            tg_send_kb_html($chat_id,$msg,$kb); return;
        }
        if ($step==='await_edit_brand') {
            $st['brand']=trim($text); $st['step']='await_edit_ig'; acc_state_set($uid,'profile',$st);
            tg_send_html($chat_id,'اینستاگرام جدید را وارد کن:'); return;
        }
        if ($step==='await_edit_ig') {
            $st['instagram']=trim($text); $st['step']='edit_confirm'; acc_state_set($uid,'profile',$st);
            $kb=kb_inline([[ kb_btn('💾 ذخیره', 'acc:c:edit:save'), kb_btn('❌ لغو', 'acc:profile:cancel') ]]);
            $msg="ویرایش مشتری:\nنام: <b>".tg_escape($st['brand'])."</b>\nاینستاگرام: <b>".tg_escape($st['instagram'])."</b>";
            tg_send_kb_html($chat_id,$msg,$kb); return;
        }
    } else {
        /* جمع‌آوری پروفایل کارمند */
        $step=$st['step']??'';
        if ($step==='await_fullname') {
            $st['full_name']=trim($text); $st['step']='await_job'; acc_state_set($uid,'profile',$st);
            tg_send_html($chat_id,'نقش/سمت کارمند را وارد کن (مثلاً: ادمین ارشد):'); return;
        }
        if ($step==='await_job') {
            $st['job']=trim($text); $st['step']='confirm'; acc_state_set($uid,'profile',$st);
            $kb=kb_inline([[ kb_btn('✅ ثبت', 'acc:e:new:save'), kb_btn('❌ لغو', 'acc:profile:cancel') ]]);
            $msg="پیش‌نویس کارمند:\nنام: <b>".tg_escape($st['full_name'])."</b>\nسمت: <b>".tg_escape($st['job'])."</b>";
            tg_send_kb_html($chat_id,$msg,$kb); return;
        }
        if ($step==='await_edit_name') {
            $st['full_name']=trim($text); $st['step']='await_edit_job'; acc_state_set($uid,'profile',$st);
            tg_send_html($chat_id,'سمت جدید را وارد کن:'); return;
        }
        if ($step==='await_edit_job') {
            $st['job']=trim($text); $st['step']='edit_confirm'; acc_state_set($uid,'profile',$st);
            $kb=kb_inline([[ kb_btn('💾 ذخیره', 'acc:e:edit:save'), kb_btn('❌ لغو', 'acc:profile:cancel') ]]);
            $msg="ویرایش کارمند:\nنام: <b>".tg_escape($st['full_name'])."</b>\nسمت: <b>".tg_escape($st['job'])."</b>";
            tg_send_kb_html($chat_id,$msg,$kb); return;
        }
    }
}

function acc_customer_save_new(int $chat_id, int $uid): void {
    $st=acc_state_get($uid,'profile'); if(!$st||($st['type']??'')!=='customer'||($st['step']??'')!=='confirm'){ tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $all=storage_load(CUSTOMERS_FILE,[]);
    $cid = (int)(time().mt_rand(100,999));
    $all[]=['chat_id'=>$cid,'brand'=>$st['brand'],'instagram'=>$st['instagram'],'active'=>true];
    storage_save(CUSTOMERS_FILE,$all);
    acc_state_clear($uid);
    tg_send_html($chat_id,"✅ مشتری «<b>".tg_escape($st['brand'])."</b>» ثبت شد.");
    acc_customers_list($chat_id,1);
}
function acc_customer_toggle(int $chat_id, int $cust_id): void {
    $all=storage_load(CUSTOMERS_FILE,[]); foreach($all as &$c) if((int)($c['chat_id']??0)===$cust_id){ $c['active']=!($c['active']??true); storage_save(CUSTOMERS_FILE,$all); break; }
    acc_customer_profile($chat_id,$cust_id);
}
function acc_customer_delete(int $chat_id, int $cust_id): void {
    $all=storage_load(CUSTOMERS_FILE,[]); $all=array_values(array_filter($all, fn($c)=>(int)($c['chat_id']??0)!==$cust_id)); storage_save(CUSTOMERS_FILE,$all);
    tg_send_html($chat_id,"🗑 مشتری حذف شد.");
    acc_customers_list($chat_id,1);
}
function acc_customer_edit_start(int $chat_id, int $uid, int $cust_id): void {
    $all=storage_load(CUSTOMERS_FILE,[]); $c=null; foreach($all as $x) if((int)($x['chat_id']??0)===$cust_id){$c=$x;break;}
    if(!$c){ tg_send_html($chat_id,'⛔️ مشتری یافت نشد.'); return; }
    acc_state_set($uid,'profile',['type'=>'customer','step'=>'await_edit_brand','cust'=>$cust_id,'brand'=>$c['brand']??'','instagram'=>$c['instagram']??'']);
    tg_send_html($chat_id,'نام جدید مشتری را وارد کن:');
}
function acc_customer_edit_save(int $chat_id, int $uid): void {
    $st=acc_state_get($uid,'profile'); if(!$st||($st['type']??'')!=='customer'||($st['step']??'')!=='edit_confirm'){ tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $all=storage_load(CUSTOMERS_FILE,[]); foreach($all as &$c) if((int)($c['chat_id']??0)===(int)$st['cust']){ $c['brand']=$st['brand']; $c['instagram']=$st['instagram']; storage_save(CUSTOMERS_FILE,$all); break; }
    acc_state_clear($uid);
    tg_send_html($chat_id,"✅ اطلاعات مشتری به‌روزرسانی شد.");
    acc_customer_profile($chat_id,(int)$st['cust']);
}

/* ===================== پرسنل (CRUD + مرخصی) ===================== */

const LEAVES_FILE = __DIR__ . '/../storage/leaves.json';

function acc_employees_list(int $chat_id, int $page=1): void {
    $all = storage_load(EMPLOYEES_FILE, []);
    if (!$all) { tg_send_kb_html($chat_id,'هنوز پرسنلی ثبت نشده.', acc_back_kb('acc:employees')); return; }
    usort($all, fn($a,$b)=>strcmp(($a['full_name']??''), ($b['full_name']??'')));
    $total=count($all); $pages=(int)ceil($total/ACC_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($all, ($page-1)*ACC_PAGE_SIZE, ACC_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $e) {
        $uid=(int)($e['chat_id']??0);
        $name=$e['full_name'] ?: '—'; $job=$e['job'] ?: '—';
        $rows[]=[ kb_btn("👤 {$name} — {$job}", "acc:e:show:{$uid}") ];
    }
    $nav=acc_paginate_kb('acc:e:list',$page,$pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows,$nav);
    $rows[]=[ kb_btn('◀️ بازگشت','acc:employees') ];
    tg_send_kb_html($chat_id,"📋 <b>فهرست پرسنل</b>", ['inline_keyboard'=>$rows]);
}
function acc_employee_profile(int $chat_id, int $uid): void {
    $emps=storage_load(EMPLOYEES_FILE,[]); $e=null; foreach($emps as $x) if((int)($x['chat_id']??0)===$uid){$e=$x;break;}
    if(!$e){ tg_send_html($chat_id,'⛔️ کارمند یافت نشد.'); return; }
    $name=$e['full_name']?:'—'; $job=$e['job']?:'—'; $active=($e['active']??true)?'فعال 🟢':'غیرفعال 🔴';

    // مرخصی‌های در انتظار
    $lv = acc_json_load(LEAVES_FILE);
    $pend = array_values(array_filter($lv, fn($r)=> (int)($r['uid']??0)===$uid && ($r['status']??'')==='pending'));
    $pendTxt = count($pend) ? " | مرخصی‌های در انتظار: ".to_persian_num(count($pend)) : "";

    $kb = kb_inline([
        [ kb_btn('✏️ ویرایش', "acc:e:edit:{$uid}"), kb_btn(($e['active']??true)?'🔒 غیرفعال':'🔓 فعال', "acc:e:toggle:{$uid}") ],
        [ kb_btn('🗑 حذف', "acc:e:del:{$uid}"), kb_btn('📝 مرخصی‌ها', "acc:e:leave:{$uid}") ],
        [ kb_btn('◀️ بازگشت','acc:e:list') ],
    ]);
    $msg="👤 <b>{$name}</b>\nسمت: <b>{$job}</b>\nوضعیت: {$active}{$pendTxt}";
    tg_send_kb_html($chat_id,$msg,$kb);
}
function acc_employee_new_start(int $chat_id, int $uid): void {
    acc_state_set($uid,'profile',['type'=>'employee','step'=>'await_fullname']);
    tg_send_kb_html($chat_id,'نام و نام‌خانوادگی کارمند را وارد کن:', acc_back_kb('acc:employees'));
}
function acc_employee_save_new(int $chat_id, int $uid): void {
    $st=acc_state_get($uid,'profile'); if(!$st||($st['type']??'')!=='employee'||($st['step']??'')!=='confirm'){ tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $emps=storage_load(EMPLOYEES_FILE,[]);
    $id=(int)(time().mt_rand(100,999));
    $emps[] = ['chat_id'=>$id,'full_name'=>$st['full_name'],'job'=>$st['job'],'active'=>true,'deductions'=>[],'bonuses'=>[]];
    storage_save(EMPLOYEES_FILE,$emps);
    acc_state_clear($uid);
    tg_send_html($chat_id,"✅ کارمند «<b>".tg_escape($st['full_name'])."</b>» ثبت شد.");
    acc_employees_list($chat_id,1);
}
function acc_employee_toggle(int $chat_id, int $uid_): void {
    $emps=storage_load(EMPLOYEES_FILE,[]); foreach($emps as &$e) if((int)($e['chat_id']??0)===$uid_){ $e['active']=!($e['active']??true); storage_save(EMPLOYEES_FILE,$emps); break; }
    acc_employee_profile($chat_id,$uid_);
}
function acc_employee_delete(int $chat_id, int $uid_): void {
    $emps=storage_load(EMPLOYEES_FILE,[]); $emps=array_values(array_filter($emps, fn($e)=>(int)($e['chat_id']??0)!==$uid_)); storage_save(EMPLOYEES_FILE,$emps);
    tg_send_html($chat_id,"🗑 کارمند حذف شد.");
    acc_employees_list($chat_id,1);
}
function acc_employee_edit_start(int $chat_id, int $uid, int $emp_uid): void {
    $emps=storage_load(EMPLOYEES_FILE,[]); $e=null; foreach($emps as $x) if((int)($x['chat_id']??0)===$emp_uid){$e=$x;break;}
    if(!$e){ tg_send_html($chat_id,'⛔️ کارمند یافت نشد.'); return; }
    acc_state_set($uid,'profile',['type'=>'employee','step'=>'await_edit_name','emp'=>$emp_uid,'full_name'=>$e['full_name']??'','job'=>$e['job']??'']);
    tg_send_html($chat_id,'نام جدید کارمند را وارد کن:');
}
function acc_employee_edit_save(int $chat_id, int $uid): void {
    $st=acc_state_get($uid,'profile'); if(!$st||($st['type']??'')!=='employee'||($st['step']??'')!=='edit_confirm'){ tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $emps=storage_load(EMPLOYEES_FILE,[]); foreach($emps as &$e) if((int)($e['chat_id']??0)===(int)$st['emp']){ $e['full_name']=$st['full_name']; $e['job']=$st['job']; storage_save(EMPLOYEES_FILE,$emps); break; }
    acc_state_clear($uid);
    tg_send_html($chat_id,"✅ اطلاعات کارمند به‌روزرسانی شد.");
    acc_employee_profile($chat_id,(int)$st['emp']);
}

/* مرخصی‌ها (درخواست‌ها و تأیید/رد توسط حسابدار) */
function acc_employee_leave_list(int $chat_id, int $emp_uid): void {
    $lv = acc_json_load(LEAVES_FILE);
    $mine = array_values(array_filter($lv, fn($r)=> (int)($r['uid']??0)===$emp_uid));
    if (!$mine) { tg_send_kb_html($chat_id,'هیچ مرخصی ثبت نشده.', acc_back_kb("acc:e:show:{$emp_uid}")); return; }
    usort($mine, fn($a,$b)=>($b['ts']??0) <=> ($a['ts']??0));
    $rows=[];
    foreach ($mine as $m) {
        $status = $m['status'] ?? 'pending';
        $rng = (string)($m['from']??'').' → '.(string)($m['to']??'');
        $rows[] = [ kb_btn("🗓 {$rng} — {$status}", "acc:e:leave:view:{$m['id']}") ];
    }
    $rows[]=[ kb_btn('◀️ بازگشت', "acc:e:show:{$emp_uid}") ];
    tg_send_kb_html($chat_id, "📝 <b>مرخصی‌های کارمند</b>", ['inline_keyboard'=>$rows]);
}
function acc_employee_leave_view(int $chat_id, int $id): void {
    $lv = acc_json_load(LEAVES_FILE);
    $rec=null; foreach ($lv as $x) if ((int)($x['id']??0)===$id) { $rec=$x; break; }
    if (!$rec) { tg_send_html($chat_id,'⛔️ مرخصی یافت نشد.'); return; }
    $name = acc_person_title_by_id((int)($rec['uid']??0),'employee');
    $msg = "📝 <b>مرخصی #".to_persian_num($id)."</b>\n"
         . "کارمند: <b>{$name}</b>\n"
        . "از: <b>".tg_escape((string)($rec['from']??'')) ."</b>\n"
        . "تا: <b>".tg_escape((string)($rec['to']??''))   ."</b>\n"
        . "علت: <b>".tg_escape((string)($rec['reason']??'')) ."</b>\n"
        . "وضعیت: <b>".tg_escape((string)($rec['status']??'pending')) ."</b>";
    $kb = kb_inline([
        [ kb_btn('✅ تایید', "acc:e:leave:approve:{$id}"), kb_btn('❌ رد', "acc:e:leave:reject:{$id}") ],
        [ kb_btn('◀️ بازگشت','acc:e:list') ],
    ]);
    tg_send_kb_html($chat_id,$msg,$kb);
}
function acc_employee_leave_set(int $chat_id, int $id, string $status): void {
    $lv = acc_json_load(LEAVES_FILE);
    foreach ($lv as &$r) if ((int)($r['id']??0)===$id) { $r['status']=$status; break; }
    acc_json_save(LEAVES_FILE,$lv);
    tg_send_html($chat_id, $status==='approved'?'✅ مرخصی تایید شد.':'❌ مرخصی رد شد.');
}

/* ===================== دفترکل (ادامه) ===================== */

function acc_cb_add_collect(int $chat_id, int $uid, string $text): void {
    $st = acc_state_get($uid,'cb'); if (!$st) return;
    $type = $st['type'] ?? 'income';
    $step = $st['step'] ?? '';
    if ($step==='await_title') {
        $st['title'] = trim($text);
        $st['step']  = 'await_amount';
        acc_state_set($uid,'cb',$st);
        tg_send_html($chat_id, "مبلغ (تومان) را وارد کن (فقط عدد).");
        return;
    }
    if ($step==='await_amount') {
        $num = (int)preg_replace('~\D+~','', $text);
        if ($num<=0) { tg_send_html($chat_id,'⛔️ مبلغ نامعتبر. دوباره بفرست.'); return; }
        $st['amount'] = $num;
        $st['step']   = 'confirm';
        acc_state_set($uid,'cb',$st);
        $kb = kb_inline([[ kb_btn('✅ تایید و ثبت', 'acc:cb:save'), kb_btn('❌ لغو', 'acc:cb:cancel') ]]);
        tg_send_kb_html($chat_id, ($type==='income'?'درآمد':'هزینه').":\nعنوان: <b>".tg_escape($st['title'])."</b>\nمبلغ: <b>".money($num)."</b>", $kb);
        return;
    }
}
function acc_cb_save_entry(int $chat_id, int $uid): void {
    $st = acc_state_get($uid,'cb'); if (!$st || ($st['step']??'')!=='confirm') { tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $arr = acc_json_load(CASHBOOK_FILE);
    $arr[] = [
        'id'     => time().mt_rand(100,999),
        'type'   => $st['type'] ?? 'income',
        'title'  => $st['title'] ?? '',
        'amount' => (int)($st['amount'] ?? 0),
        'by'     => $uid,
        'ts'     => time()
    ];
    acc_json_save(CASHBOOK_FILE,$arr);
    acc_state_clear($uid);
    tg_send_kb_html($chat_id, "✅ ثبت شد.", acc_back_kb('acc:cashbook'));
}
function acc_cb_list(int $chat_id, int $page=1, string $filter='all'): void {
    $arr = acc_json_load(CASHBOOK_FILE);
    if ($filter!=='all') $arr = array_values(array_filter($arr, fn($x)=> ($x['type']??'')===$filter));
    if (!$arr) { tg_send_kb_html($chat_id, 'رکوردی یافت نشد.', acc_back_kb('acc:back')); return; }
    usort($arr, fn($a,$b)=>($b['ts']??0) <=> ($a['ts']??0));
    $total=count($arr); $pages=(int)ceil($total/ACC_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($arr, ($page-1)*ACC_PAGE_SIZE, ACC_PAGE_SIZE);

    $lines=[];
    $sumInc=0; $sumExp=0;
    foreach ($slice as $x) {
        $isInc = ($x['type']==='income'); 
        if ($isInc) $sumInc += (int)$x['amount']; else $sumExp += (int)$x['amount'];
        $lines[] = ($isInc?'🔹':'🔸')." ".tg_escape($x['title'])." — ".money((int)$x['amount'])." — ".jdate('Y/m/d H:i',$x['ts']);
    }
    $msg = "<b>لیست ".($filter==='all'?'درآمد/هزینه':'فقط '.($filter==='income'?'درآمد':'هزینه'))."</b>\n"
         . implode("\n",$lines)."\n\n"
         . "جمع این صفحه → درآمد: <b>".money($sumInc)."</b> | هزینه: <b>".money($sumExp)."</b>";

    $rows = [];
    $nav=acc_paginate_kb("acc:cb:list:{$filter}", $page,$pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows,$nav);
    $rows[]=[ kb_btn('◀️ بازگشت','acc:cashbook') ];
    tg_send_kb_html($chat_id, $msg, ['inline_keyboard'=>$rows]);
}
/* ===================== صورتجلسه (Minutes) ===================== */

function acc_minutes_new_start(int $chat_id, int $uid): void {
    acc_state_set($uid,'min',['step'=>'title']);
    tg_send_kb_html($chat_id, "🔹 عنوان صورتجلسه را وارد کن:", acc_back_kb('acc:minutes'));
}
function acc_minutes_collect(int $chat_id, int $uid, string $text): void {
    $st = acc_state_get($uid,'min'); if (!$st) return;
    $step = $st['step'] ?? '';
    if ($step==='title') {
        $st['title'] = trim($text);
        $st['step']  = 'about';
        acc_state_set($uid,'min',$st);
        tg_send_html($chat_id, "🔹 موضوع/دربارهٔ صورتجلسه را وارد کن (مثلاً: تخلف/هشدار/توافق مالی/تغییر تیم):");
        return;
    }
    if ($step==='about') {
        $st['about'] = trim($text);
        $st['step']  = 'parties';
        acc_state_set($uid,'min',$st);
        tg_send_html($chat_id, "🔹 طرف‌های ذی‌نفع را بنویس (نام‌ها با کاما جدا؛ فقط نام‌ها، آی‌دی عددی نشان داده نمی‌شود):");
        return;
    }
    if ($step==='parties') {
        $st['parties'] = trim($text);
        $st['step']    = 'body';
        acc_state_set($uid,'min',$st);
        tg_send_html($chat_id, "🔹 متن رسمی و اداری صورتجلسه را کامل بنویس (شرح، مستندات، تکالیف، مهلت‌ها):");
        return;
    }
    if ($step==='body') {
        $st['body'] = trim($text);
        $st['step'] = 'confirm';
        acc_state_set($uid,'min',$st);
        $kb = kb_inline([[ kb_btn('✅ ثبت و ارسال برای امضا', 'acc:min:new:save'), kb_btn('❌ لغو', 'acc:min:new:cancel') ]]);
        $preview = "📝 پیش‌نویس صورتجلسه:\n"
                 . "عنوان: <b>".tg_escape($st['title'])."</b>\n"
                 . "موضوع: <b>".tg_escape($st['about'])."</b>\n"
                 . "طرف‌ها: <b>".tg_escape($st['parties'])."</b>\n"
                 . "متن:\n".tg_escape($st['body']);
        tg_send_kb_html($chat_id, $preview, $kb);
        return;
    }
}

function acc_minutes_save(int $chat_id, int $uid): void {
    $st = acc_state_get($uid,'min'); 
    if (!$st || ($st['step']??'')!=='confirm') { tg_send_html($chat_id,'⛔️ مرحله نامعتبر.'); return; }
    $arr = acc_json_load(MINUTES_FILE);
    $id  = (int)(time().mt_rand(100,999));
    $rec = [
        'id'      => $id,
        'title'   => $st['title'],
        'about'   => $st['about'],
        'parties' => $st['parties'],
        'body'    => $st['body'],
        'status'  => 'pending', // pending → approved/rejected
        'by'      => $uid,
        'ts'      => time(),
        'attachments'=>[]
    ];
    $arr[] = $rec;
    acc_json_save(MINUTES_FILE,$arr);
    acc_state_clear($uid);

    // ارسال برای امضا به مدیرعامل
    $kb_ceo = kb_inline([
        [ kb_btn('✅ تایید و امضا', "acc:min:approve:{$id}"), kb_btn('❌ رد', "acc:min:reject:{$id}") ],
    ]);
    tg_send_kb_html(CEO_UID, "📝 <b>صورتجلسه جدید جهت امضا</b>\n#".to_persian_num($id)." — ".tg_escape($rec['title'])."\nموضوع: ".tg_escape($rec['about'])."\nطرف‌ها: ".tg_escape($rec['parties'])."\n\nمتن:\n".tg_escape($rec['body']), $kb_ceo);

    tg_send_kb_html($chat_id, "✅ صورتجلسه #".to_persian_num($id)." ثبت و برای امضا ارسال شد.", acc_back_kb('acc:minutes'));
}

function acc_minutes_list(int $chat_id, string $filter='all', int $page=1): void {
    $arr = acc_json_load(MINUTES_FILE);
    if ($filter!=='all') $arr = array_values(array_filter($arr, fn($x)=> ($x['status']??'')===$filter));
    if (!$arr) { tg_send_kb_html($chat_id,'چیزی یافت نشد.', acc_back_kb('acc:minutes')); return; }
    usort($arr, fn($a,$b)=>($b['ts']??0) <=> ($a['ts']??0));
    $total=count($arr); $pages=(int)ceil($total/ACC_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($arr, ($page-1)*ACC_PAGE_SIZE, ACC_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $r) {
        $id=(int)$r['id']; $ttl=$r['title']??'—'; $st=$r['status']??'pending';
        $rows[]=[ kb_btn("📝 #".to_persian_num($id)." — {$st} — ".tg_escape($ttl), "acc:min:view:{$id}") ];
    }
    $nav=acc_paginate_kb("acc:min:list:{$filter}", $page,$pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows,$nav);
    $rows[]=[ kb_btn('◀️ بازگشت','acc:minutes') ];
    $title = "📝 <b>صورتجلسه‌ها</b> — ".($filter==='all'?'همه':($filter==='pending'?'در انتظار امضا':($filter==='approved'?'ابلاغ‌شده':'رد شده')));
    tg_send_kb_html($chat_id, $title, ['inline_keyboard'=>$rows]);
}

function acc_minutes_view(int $chat_id, int $id): void {
    $arr = acc_json_load(MINUTES_FILE);
    $rec=null; foreach ($arr as $x) if ((int)($x['id']??0)===$id) {$rec=$x; break;}
    if (!$rec) { tg_send_html($chat_id,'⛔️ صورتجلسه یافت نشد.'); return; }
    $msg = "📝 <b>صورتجلسه #".to_persian_num($id)."</b>\n"
         . "عنوان: <b>".tg_escape((string)$rec['title'])."</b>\n"
         . "موضوع: <b>".tg_escape((string)$rec['about'])."</b>\n"
         . "طرف‌ها: <b>".tg_escape((string)$rec['parties'])."</b>\n"
         . "وضعیت: <b>".tg_escape((string)($rec['status']??'pending'))."</b>\n\n"
         . "متن:\n".tg_escape((string)$rec['body']);
    $kb = kb_inline([
        [ kb_btn('🖨 خروجی PDF/HTML', "acc:min:export:{$id}") ],
        [ kb_btn('◀️ بازگشت','acc:min:list:all') ],
    ]);
    tg_send_kb_html($chat_id,$msg,$kb);
}
function acc_minutes_export(int $chat_id, int $id): void {
    $arr = acc_json_load(MINUTES_FILE);
    $rec=null; foreach ($arr as $x) if ((int)($x['id']??0)===$id) {$rec=$x; break;}
    if (!$rec) { tg_send_html($chat_id,'⛔️ صورتجلسه یافت نشد.'); return; }

    $html = "<html><head><meta charset='utf-8'><style>
        body{font-family:Tahoma, sans-serif;direction:rtl;line-height:1.9}
        h1{font-size:18px;margin:0 0 10px}
        .sec{margin:10px 0;padding:10px;border:1px solid #ddd;border-radius:8px}
        .muted{color:#666}
    </style></head><body>
    <h1>صورتجلسه شماره ".to_persian_num((string)$rec['id'])."</h1>
    <div class='sec'><b>عنوان:</b> ".tg_escape((string)$rec['title'])."</div>
    <div class='sec'><b>موضوع:</b> ".tg_escape((string)$rec['about'])."</div>
    <div class='sec'><b>طرف‌ها:</b> ".tg_escape((string)$rec['parties'])."</div>
    <div class='sec'><b>متن:</b><br>".nl2br(tg_escape((string)$rec['body']))."</div>
    <div class='sec muted'>وضعیت: ".tg_escape((string)($rec['status']??'pending'))." | تاریخ: ".jdate('Y/m/d H:i',(int)($rec['ts']??time()))."</div>
    <div class='sec'>امضا: مدیرعامل / حسابدار</div>
    </body></html>";

    if (class_exists('Dompdf\Dompdf')) {
        if (!is_dir(EXPORT_DIR)) @mkdir(EXPORT_DIR, 0775, true);
        $file = EXPORT_DIR . "/minute_{$id}.pdf";
        $dompdf = new Dompdf\Dompdf(); $dompdf->loadHtml($html, 'UTF-8'); $dompdf->setPaper('A4'); $dompdf->render();
        @file_put_contents($file, $dompdf->output());
        tg_send_html($chat_id, "✅ خروجی PDF آماده شد.\n<code>{$file}</code>");
    } else {
        tg_send_html($chat_id, "ℹ️ PDF فعال نیست؛ نسخهٔ HTML قابل‌پرینت:\n".$html);
    }
}

/* امضا توسط مدیرعامل (از طریق همین ربات) */
function acc_minutes_set_status(int $chat_id, int $id, string $status): void {
    $arr = acc_json_load(MINUTES_FILE);
    $ok=false;
    foreach ($arr as &$r) if ((int)($r['id']??0)===$id) { $r['status']=$status; $ok=true; break; }
    if ($ok) acc_json_save(MINUTES_FILE,$arr);
    tg_send_html($chat_id, $ok? ($status==='approved'?'✅ ابلاغ شد.':'❌ رد شد.') : '⛔️ یافت نشد.');
}

/* ===================== درخواست نقش (Onboarding) ===================== */

function acc_roles_inbox(int $chat_id, int $page=1): void {
    $rq = acc_json_load(ROLE_REQUESTS_FILE);
    if (!$rq) { tg_send_kb_html($chat_id,'هیچ درخواستی نیست.', acc_back_kb('acc:back')); return; }
    usort($rq, fn($a,$b)=>($b['ts']??0) <=> ($a['ts']??0));
    $total=count($rq); $pages=(int)ceil($total/ACC_PAGE_SIZE);
    $page=max(1,min($pages,$page)); $slice=array_slice($rq, ($page-1)*ACC_PAGE_SIZE, ACC_PAGE_SIZE);

    $rows=[];
    foreach ($slice as $r) {
        $id=(int)$r['id']; $name=$r['name']??'—'; $role=$r['role']??'—';
        $rows[]=[ kb_btn("📬 #".to_persian_num($id)." — {$name} — {$role}", "acc:role:view:{$id}") ];
    }
    $nav=acc_paginate_kb('acc:roles',$page,$pages)['inline_keyboard']??[];
    if ($nav) $rows=array_merge($rows,$nav);
    $rows[]=[ kb_btn('◀️ بازگشت','acc:back') ];
    tg_send_kb_html($chat_id, "📬 <b>درخواست‌های نقش</b>", ['inline_keyboard'=>$rows]);
}

function acc_role_view(int $chat_id, int $id): void {
    $rq = acc_json_load(ROLE_REQUESTS_FILE);
    $rec=null; foreach ($rq as $x) if ((int)($x['id']??0)===$id){$rec=$x;break;}
    if (!$rec) { tg_send_html($chat_id,'⛔️ درخواست یافت نشد.'); return; }
    $msg = "📬 <b>درخواست #".to_persian_num($id)."</b>\n"
         . "نام: <b>".tg_escape((string)$rec['name'])."</b>\n"
         . "نقش موردنظر: <b>".tg_escape((string)$rec['role'])."</b>\n"
         . "توضیحات: ".tg_escape((string)($rec['note']??'—'));
    $kb = kb_inline([
        [ kb_btn('✅ تشکیل پرونده', "acc:role:approve:{$id}"), kb_btn('❌ رد', "acc:role:reject:{$id}") ],
        [ kb_btn('◀️ بازگشت','acc:roles') ],
    ]);
    tg_send_kb_html($chat_id,$msg,$kb);
}

function acc_role_approve(int $chat_id, int $id): void {
    $rq = acc_json_load(ROLE_REQUESTS_FILE);
    $rec=null; $left=[];
    foreach ($rq as $x) {
        if ((int)($x['id']??0)===$id) $rec=$x; else $left[]=$x;
    }
    if (!$rec) { tg_send_html($chat_id,'⛔️ درخواست یافت نشد.'); return; }

    // بر اساس نقش، پرونده‌ی مشتری یا کارمند بساز
    $role = mb_strtolower((string)($rec['role']??''));
    if (str_contains($role,'مشتری') || str_contains($role,'customer')) {
        $all = storage_load(CUSTOMERS_FILE, []);
        $cid = (int)(time().mt_rand(100,999));
        $all[]=['chat_id'=>$cid,'brand'=>$rec['name'],'instagram'=>$rec['instagram']??'-','active'=>true];
        storage_save(CUSTOMERS_FILE,$all);
        tg_send_html($chat_id, "✅ پروندهٔ مشتری «<b>".tg_escape((string)$rec['name'])."</b>» ساخته شد.");
    } else {
        $emps= storage_load(EMPLOYEES_FILE, []);
        $eid = (int)(time().mt_rand(100,999));
        $emps[]=['chat_id'=>$eid,'full_name'=>$rec['name'],'job'=>$rec['role'],'active'=>true,'deductions'=>[],'bonuses'=>[]];
        storage_save(EMPLOYEES_FILE,$emps);
        tg_send_html($chat_id, "✅ پروندهٔ پرسنل «<b>".tg_escape((string)$rec['name'])."</b>» ساخته شد.");
    }
    acc_json_save(ROLE_REQUESTS_FILE, $left);
    acc_roles_inbox($chat_id,1);
}
function acc_role_reject(int $chat_id, int $id): void {
    $rq = acc_json_load(ROLE_REQUESTS_FILE);
    $rq = array_values(array_filter($rq, fn($x)=> (int)($x['id']??0)!==$id));
    acc_json_save(ROLE_REQUESTS_FILE,$rq);
    tg_send_html($chat_id, "❌ درخواست رد شد.");
    acc_roles_inbox($chat_id,1);
}

/* ===================== گزارش‌ها ===================== */

function acc_reports_menu(int $chat_id): void {
    $kb = kb_inline([
        [ kb_btn('💰 درآمد/هزینه (۳۰ روز اخیر)', 'acc:rep:cash:30'), kb_btn('🧾 مطالبات باز (Aging)', 'acc:rep:aging') ],
        [ kb_btn('🏢 بدهکاران برتر', 'acc:rep:debtors'), kb_btn('📈 عملکرد ماه جاری', 'acc:rep:month') ],
        [ kb_btn('◀️ بازگشت','acc:back') ],
    ]);
    tg_send_kb_html($chat_id, "📊 <b>گزارش‌ها</b>", $kb);
}
function acc_report_cash_30(int $chat_id): void {
    $arr = acc_json_load(CASHBOOK_FILE);
    $since = time()-30*24*3600;
    $a = array_filter($arr, fn($x)=> (int)($x['ts']??0) >= $since);
    $inc = array_sum(array_map(fn($x)=> $x['type']==='income'?(int)$x['amount']:0, $a));
    $exp = array_sum(array_map(fn($x)=> $x['type']==='expense'?(int)$x['amount']:0, $a));
    $msg = "💰 <b>۳۰ روز اخیر</b>\n"
         . "درآمد: <b>".money($inc)."</b>\n"
         . "هزینه: <b>".money($exp)."</b>\n"
         . "تراز خالص: <b>".money($inc-$exp)."</b>";
    tg_send_kb_html($chat_id,$msg, acc_back_kb('acc:reports'));
}
function acc_report_aging(int $chat_id): void {
    $invs = inv_list();
    $open = array_values(array_filter($invs, fn($i)=> in_array(($i['status']??'unpaid'), ['unpaid','partial'], true)));
    if (!$open) { tg_send_kb_html($chat_id,'مطالبهٔ بازی وجود ندارد.', acc_back_kb('acc:reports')); return; }
    usort($open, fn($a,$b)=>($a['created_ts']??0) <=> ($b['created_ts']??0));

    $lines=[];
    foreach ($open as $i) {
        $id=(int)$i['id']; $cust=acc_person_title_by_id((int)($i['customer_chat_id']??0),'customer');
        $remain = (int)(inv_remain_info($id)['remain']??0);
        $age = (int)floor((time() - (int)($i['created_ts']??time()))/86400);
        $lines[] = "🧾 #".to_persian_num($id)." — {$cust} — مانده: ".money($remain)." — عمر: ".to_persian_num($age)." روز";
    }
    $msg = "🧾 <b>گزارش Aging مطالبات</b>\n".implode("\n",$lines);
    tg_send_kb_html($chat_id,$msg, acc_back_kb('acc:reports'));
}
function acc_report_top_debtors(int $chat_id): void {
    $invs = inv_list();
    $open = array_values(array_filter($invs, fn($i)=> in_array(($i['status']??'unpaid'), ['unpaid','partial'], true)));
    $bucket = [];
    foreach ($open as $i) {
        $cid=(int)($i['customer_chat_id']??0);
        $remain=(int)(inv_remain_info((int)$i['id'])['remain']??0);
        $bucket[$cid] = ($bucket[$cid]??0) + $remain;
    }
    arsort($bucket);
    $lines=[];
    foreach (array_slice($bucket,0,10,true) as $cid=>$sum) {
        $lines[] = "🏢 ".acc_person_title_by_id((int)$cid,'customer')." — مانده: <b>".money((int)$sum)."</b>";
    }
    tg_send_kb_html($chat_id, "🏢 <b>بدهکاران برتر</b>\n".implode("\n",$lines), acc_back_kb('acc:reports'));
}
function acc_report_month(int $chat_id): void {
    $start = strtotime(date('Y-m-01 00:00:00'));
    $end   = strtotime(date('Y-m-t 23:59:59'));
    $cb = acc_json_load(CASHBOOK_FILE);
    $a = array_filter($cb, fn($x)=> (int)($x['ts']??0) >= $start && (int)($x['ts']??0) <= $end);
    $inc = array_sum(array_map(fn($x)=> $x['type']==='income'?(int)$x['amount']:0, $a));
    $exp = array_sum(array_map(fn($x)=> $x['type']==='expense'?(int)$x['amount']:0, $a));
    $msg = "📈 <b>ماه جاری</b>\n"
         . "درآمد: <b>".money($inc)."</b>\n"
         . "هزینه: <b>".money($exp)."</b>\n"
         . "تراز: <b>".money($inc-$exp)."</b>";
    tg_send_kb_html($chat_id,$msg, acc_back_kb('acc:reports'));
}

/* ===================== آغازگرهای دفترکل ===================== */
function acc_cb_add_start(int $chat_id, int $uid, string $type): void {
    acc_state_set($uid,'cb',['type'=>$type,'step'=>'await_title']);
    tg_send_kb_html($chat_id, "عنوان ".($type==='income'?'درآمد':'هزینه')." را وارد کن:", acc_back_kb('acc:cashbook'));
}

/* ===================== مسیرهای کال‌بک (Routing) — تکمیلی ===================== */

if (!function_exists('accountant_handle_callback_more')) {
    function accountant_handle_callback_more(int $chat_id, int $user_id, string $data): bool {
        if (!acc_is_accountant_or_ceo($user_id) || !str_starts($data,'acc:')) return false;

        /* فاکتورها */
        if (preg_match('~^acc:inv:view:(\d+)$~',$data,$m)) { acc_invoice_view($chat_id,(int)$m[1]); return true; }
        if ($data==='acc:inv:new') { acc_invoice_new_start($chat_id,$user_id); return true; }
        if (preg_match('~^acc:inv:new:c:(\d+)$~',$data,$m)) { $st=['step'=>'await_title','customer'=>(int)$m[1]]; acc_state_set($user_id,'inv',$st); tg_send_html($chat_id,'🔹 عنوان فاکتور را وارد کن:'); return true; }
        if ($data==='acc:inv:new:save') { acc_invoice_create_save($chat_id,$user_id); return true; }
        if ($data==='acc:inv:new:cancel') { acc_state_clear($user_id); tg_send_html($chat_id,'لغو شد.'); acc_invoices_list($chat_id,1,null); return true; }
        if (preg_match('~^acc:inv:for:(\d+)$~',$data,$m)) { acc_invoices_list($chat_id,1,(int)$m[1]); return true; }
        if (preg_match('~^acc:inv:p:(\d+)$~',$data,$m)) { acc_invoices_list($chat_id,(int)$m[1],null); return true; }
        if (preg_match('~^acc:inv:recv:(\d+)$~',$data,$m)) { acc_invoice_receive_prompt($chat_id,$user_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:inv:settle:(\d+)$~',$data,$m)) { acc_invoice_settle_all($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:inv:latefee:(\d+)$~',$data,$m)) { acc_invoice_apply_latefee($chat_id,(int)$m[1]); return true; }

        /* قراردادها */
        if (preg_match('~^acc:con:view:(\d+)$~',$data,$m)) { acc_contract_view($chat_id,(int)$m[1]); return true; }
        if ($data==='acc:con:new') { acc_contract_new_start($chat_id,$user_id); return true; }
        if (preg_match('~^acc:con:new:c:(\d+)$~',$data,$m)) { acc_state_set($user_id,'contract',['step'=>'await_title','customer'=>(int)$m[1]]); tg_send_html($chat_id,'🔹 عنوان قرارداد را وارد کن:'); return true; }
        if ($data==='acc:con:new:save') { acc_contract_create_save($chat_id,$user_id); return true; }
        if ($data==='acc:con:new:cancel') { acc_state_clear($user_id); tg_send_html($chat_id,'لغو شد.'); acc_contracts_list($chat_id,1,null); return true; }
        if (preg_match('~^acc:con:for:(\d+)$~',$data,$m)) { acc_contracts_list($chat_id,1,(int)$m[1]); return true; }
        if (preg_match('~^acc:con:p:(\d+)$~',$data,$m)) { acc_contracts_list($chat_id,(int)$m[1],null); return true; }
        if (preg_match('~^acc:con:export:(\d+)$~',$data,$m)) { acc_contract_export($chat_id,(int)$m[1]); return true; }

        /* مشتریان */
        if ($data==='acc:c:list') { acc_customers_list($chat_id,1); return true; }
        if (preg_match('~^acc:c:list:p:(\d+)$~',$data,$m)) { acc_customers_list($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:c:show:(\d+)$~',$data,$m)) { acc_customer_profile($chat_id,(int)$m[1]); return true; }
        if ($data==='acc:c:new') { acc_customer_new_start($chat_id,$user_id); return true; }
        if ($data==='acc:c:new:save') { acc_customer_save_new($chat_id,$user_id); return true; }
        if ($data==='acc:profile:cancel') { acc_state_clear($user_id); tg_send_html($chat_id,'لغو شد.'); accountant_menu_main($chat_id); return true; }
        if (preg_match('~^acc:c:toggle:(\d+)$~',$data,$m)) { acc_customer_toggle($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:c:del:(\d+)$~',$data,$m)) { acc_customer_delete($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:c:edit:(\d+)$~',$data,$m)) { acc_customer_edit_start($chat_id,$user_id,(int)$m[1]); return true; }
        if ($data==='acc:c:edit:save') { acc_customer_edit_save($chat_id,$user_id); return true; }
        if ($data==='acc:c:search') { tg_send_html($chat_id,"دستور سریع جستجو: \n<code>جستجو نام_مشتری</code>\nمثال: <code>جستجو البرز</code>"); return true; }

        /* پرسنل */
        if ($data==='acc:e:list') { acc_employees_list($chat_id,1); return true; }
        if (preg_match('~^acc:e:list:p:(\d+)$~',$data,$m)) { acc_employees_list($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:e:show:(\d+)$~',$data,$m)) { acc_employee_profile($chat_id,(int)$m[1]); return true; }
        if ($data==='acc:e:new') { acc_employee_new_start($chat_id,$user_id); return true; }
        if ($data==='acc:e:new:save') { acc_employee_save_new($chat_id,$user_id); return true; }
        if (preg_match('~^acc:e:toggle:(\d+)$~',$data,$m)) { acc_employee_toggle($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:e:del:(\d+)$~',$data,$m)) { acc_employee_delete($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:e:edit:(\d+)$~',$data,$m)) { acc_employee_edit_start($chat_id,$user_id,(int)$m[1]); return true; }
        if ($data==='acc:e:edit:save') { acc_employee_edit_save($chat_id,$user_id); return true; }
        if (preg_match('~^acc:e:leave:(\d+)$~',$data,$m)) { acc_employee_leave_list($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:e:leave:view:(\d+)$~',$data,$m)) { acc_employee_leave_view($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:e:leave:(approve|reject):(\d+)$~',$data,$m)) { acc_employee_leave_set($chat_id,(int)$m[2], $m[1]==='approve'?'approved':'rejected'); return true; }
        if ($data==='acc:e:search') { tg_send_html($chat_id,"دستور سریع جستجو: \n<code>جستجو نام_پرسنل</code>\nمثال: <code>جستجو آیدا لطفی</code>"); return true; }

        /* صورتجلسه */
        if ($data==='acc:min:new') { acc_minutes_new_start($chat_id,$user_id); return true; }
        if (preg_match('~^acc:min:list:(all|pending|approved|rejected)$~',$data,$m)) { acc_minutes_list($chat_id,$m[1],1); return true; }
        if (preg_match('~^acc:min:list:(all|pending|approved|rejected):p:(\d+)$~',$data,$m)) { acc_minutes_list($chat_id,$m[1],(int)$m[2]); return true; }
        if (preg_match('~^acc:min:view:(\d+)$~',$data,$m)) { acc_minutes_view($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:min:export:(\d+)$~',$data,$m)) { acc_minutes_export($chat_id,(int)$m[1]); return true; }
        // امضا توسط مدیرعامل
        if (preg_match('~^acc:min:(approve|reject):(\d+)$~',$data,$m)) { acc_minutes_set_status($chat_id,(int)$m[2], $m[1]==='approve'?'approved':'rejected'); return true; }

        /* دفترکل */
        if ($data==='acc:cb:add:income') { acc_cb_add_start($chat_id,$user_id,'income'); return true; }
        if ($data==='acc:cb:add:expense'){ acc_cb_add_start($chat_id,$user_id,'expense'); return true; }
        if (preg_match('~^acc:cb:list:(all|income|expense)$~',$data,$m)) { acc_cb_list($chat_id,1,$m[1]); return true; }
        if (preg_match('~^acc:cb:list:(all|income|expense):p:(\d+)$~',$data,$m)) { acc_cb_list($chat_id,(int)$m[2],$m[1]); return true; }
        if ($data==='acc:cb:save') { acc_cb_save_entry($chat_id,$user_id); return true; }
        if ($data==='acc:cb:cancel') { acc_state_clear($user_id); tg_send_html($chat_id,'لغو شد.'); acc_cashbook_menu($chat_id); return true; }

        /* نقش‌ها */
        if (preg_match('~^acc:role:view:(\d+)$~',$data,$m)) { acc_role_view($chat_id,(int)$m[1]); return true; }
        if (preg_match('~^acc:role:(approve|reject):(\d+)$~',$data,$m)) { $m[1]==='approve' ? acc_role_approve($chat_id,(int)$m[2]) : acc_role_reject($chat_id,(int)$m[2]); return true; }
        if (preg_match('~^acc:roles:p:(\d+)$~',$data,$m)) { acc_roles_inbox($chat_id,(int)$m[1]); return true; }

        /* گزارش‌ها */
        if ($data==='acc:rep:aging') { acc_report_aging($chat_id); return true; }
        if ($data==='acc:rep:debtors') { acc_report_top_debtors($chat_id); return true; }
        if ($data==='acc:rep:month') { acc_report_month($chat_id); return true; }
        if ($data==='acc:rep:cash:30') { acc_report_cash_30($chat_id); return true; }

        return false;
    }
}

/* الحاق‌ی روتر تکمیلی به روتر اصلی حسابداری */
if (!function_exists('accountant_handle_callback')) {
    // در بخش 1 تعریف شده؛ اما اگر نبود، ایمن‌سازی
    function accountant_handle_callback(int $chat_id, int $user_id, string $data): bool {
        if (!acc_is_accountant_or_ceo($user_id) || !str_starts($data,'acc:')) return false;
        // مسیرهای پایه (بخش 1) را دوباره پوشش می‌دهیم
        if ($data==='acc:back')       { accountant_menu_main($chat_id); return true; }
        if ($data==='acc:invoices')   { acc_invoices_list($chat_id,1,null); return true; }
        if ($data==='acc:contracts')  { acc_contracts_list($chat_id,1,null); return true; }
        if ($data==='acc:customers')  { acc_customers_menu($chat_id); return true; }
        if ($data==='acc:employees')  { acc_employees_menu($chat_id); return true; }
        if ($data==='acc:minutes')    { acc_minutes_menu($chat_id); return true; }
        if ($data==='acc:cashbook')   { acc_cashbook_menu($chat_id); return true; }
        if ($data==='acc:roles')      { acc_roles_inbox($chat_id,1); return true; }
        if ($data==='acc:reports')    { acc_reports_menu($chat_id); return true; }
        if ($data==='acc:exec')       { acc_exec_summary($chat_id); return true; }

        return accountant_handle_callback_more($chat_id,$user_id,$data);
    }
} else {
    // اگر نسخه بخش 1 وجود دارد، اینجا اتصال مسیرهای تکمیلی را تضمین می‌کنیم:
    if (!function_exists('___acc_router_patch_once')) {
        function ___acc_router_patch_once(): void { /* علامت فقط یکبار */ }
        // هیچ تغییر اجرایی لازم نیست؛ accountant_handle_callback_more در انتها توسط index.php فراخوانی می‌شود.
    }
}
