<?php
declare(strict_types=1);

/**
 * Vision Media Bot — approvals.php (Part 1/7)
 * مدیریت تاییدیه‌ها (جریمه، مرخصی، قرارداد و ...)
 */

const APPROVALS_FILE = __DIR__ . '/../storage/approvals.json';

// --- Load / Save ---
function approvals_all(): array {
    return storage_load(APPROVALS_FILE);
}
function approvals_save(array $all): void {
    storage_save(APPROVALS_FILE, $all);
}
// --- Create new approval ---
function approval_create(string $type, int $target_uid, array $data): string {
    $all = approvals_all();
    $id  = new_id('appr');
    $row = [
        'id'      => $id,
        'type'    => $type,       // penalty, leave, contract, invoice ...
        'target'  => $target_uid, // پرسنل یا مشتری
        'data'    => $data,
        'status'  => 'pending',   // pending / approved / rejected
        'history' => [],
        'time'    => time(),
    ];
    $all[] = $row;
    approvals_save($all);
    return $id;
}

// --- Get one ---
function approval_get(string $id): ?array {
    $all = approvals_all();
    foreach ($all as $a) {
        if ($a['id'] === $id) return $a;
    }
    return null;
}
// --- Update status ---
function approval_set_status(string $id, string $status, int $by_uid, string $note=''): void {
    $all = approvals_all();
    foreach ($all as &$a) {
        if ($a['id'] === $id) {
            $a['status'] = $status;
            $a['history'][] = [
                'by'   => $by_uid,
                'note' => $note,
                'time' => time(),
                'st'   => $status,
            ];
        }
    }
    approvals_save($all);
}

// --- Approvals by target ---
function approvals_by_target(int $uid): array {
    $all = approvals_all();
    return array_values(array_filter($all, fn($a) => (int)$a['target'] === $uid));
}
// --- Approvals by type ---
function approvals_by_type(string $type): array {
    $all = approvals_all();
    return array_values(array_filter($all, fn($a) => ($a['type'] ?? '') === $type));
}

// --- Pending approvals ---
function approvals_pending(): array {
    $all = approvals_all();
    return array_values(array_filter($all, fn($a) => ($a['status'] ?? '') === 'pending'));
}

// --- Add note ---
function approval_add_note(string $id, int $by_uid, string $note): void {
    $all = approvals_all();
    foreach ($all as &$a) {
        if ($a['id'] === $id) {
            $a['history'][] = [
                'by'   => $by_uid,
                'note' => $note,
                'time' => time(),
                'st'   => $a['status'],
            ];
        }
    }
    approvals_save($all);
}
// --- Workflow helpers ---
function approval_request_penalty(int $uid, int $by_manager, int $amount, string $reason): string {
    $data = [
        'amount' => $amount,
        'reason' => $reason,
        'by'     => $by_manager,
    ];
    return approval_create('penalty', $uid, $data);
}

function approval_request_leave(int $uid, string $from, string $to, string $reason): string {
    $data = [
        'from'   => $from,
        'to'     => $to,
        'reason' => $reason,
    ];
    return approval_create('leave', $uid, $data);
}

function approval_request_contract(int $cust_id, array $contract): string {
    return approval_create('contract', $cust_id, $contract);
}
// --- Status checks ---
function approval_is_pending(array $a): bool {
    return ($a['status'] ?? '') === 'pending';
}
function approval_is_approved(array $a): bool {
    return ($a['status'] ?? '') === 'approved';
}
function approval_is_rejected(array $a): bool {
    return ($a['status'] ?? '') === 'rejected';
}

// --- Short label ---
function approval_label(array $a): string {
    $t = isset($a['type']) ? $a['type'] : '';
    $map = [
        'penalty'  => 'جریمه',
        'leave'    => 'مرخصی',
        'contract' => 'قرارداد',
        'invoice'  => 'فاکتور',
    ];
    return isset($map[$t]) ? $map[$t] : $t;
}

// --- Notifications ---
function approval_notify(string $id, string $msg): void {
    $a = approval_get($id);
    if (!$a) return;
    $target = (int)($a['target'] ?? 0);
    if ($target) tg_send_html($target, $msg);

    // اطلاع به مدیرعامل و حسابداری
    tg_send_html(CEO_ID, "🔔 {$msg}");
    $accs = users_by_role('ACCOUNTING');
    foreach ($accs as $ac) {
        tg_send_html((int)$ac['id'], "🔔 {$msg}");
    }
}

// --- پایان approvals.php ---
