<?php
declare(strict_types=1);

/**
 * Vision Media Bot — tasks.php (Part 1/7)
 * مدیریت وظایف (Tasks)
 */

const TASKS_FILE = __DIR__ . '/../storage/tasks.json';

// --- Load/Save ---
function tasks_all(): array {
    return storage_load(TASKS_FILE);
}
function tasks_save(array $all): void {
    storage_save(TASKS_FILE, $all);
}
// --- Create new task ---
function task_create(int $uid, string $title, int $deadline, string $by_role='SYSTEM'): string {
    $all = tasks_all();
    $id  = new_id('task');
    $row = [
        'id'       => $id,
        'uid'      => $uid,
        'title'    => $title,
        'deadline' => $deadline,
        'by'       => $by_role,
        'done'     => null,
        'warn5h'   => false,
        'warn1h'   => false,
        'time'     => time(),
    ];
    $all[] = $row;
    tasks_save($all);

    tg_send_html($uid,
        "📌 یک وظیفه جدید برایت ثبت شد:\n"
        . "➖ {$title}\n"
        . "⏰ مهلت: " . dt_jalali($deadline)
    );
    return $id;
}

// --- Get by user ---
function tasks_for_user(int $uid): array {
    return array_values(array_filter(tasks_all(), fn($t) => (int)$t['uid'] === $uid));
}
// --- Mark done ---
function task_done(int $uid, string $tid): void {
    $all = tasks_all();
    foreach ($all as &$t) {
        if ($t['id'] === $tid && (int)$t['uid'] === $uid) {
            $t['done'] = time();
        }
    }
    tasks_save($all);

    tg_send_success($uid, "آفرین 👏 وظیفه «{$tid}» رو به موقع انجام دادی.");
}

// --- Overdue check ---
function task_check_overdue(): void {
    $all = tasks_all();
    foreach ($all as $t) {
        if (empty($t['done']) && $t['deadline'] < time()) {
            pen_auto_for_task_deadline((int)$t['uid'], $t['title']);
        }
    }
}
// --- Reminders ---
function task_send_reminders(): void {
    $all = tasks_all();
    foreach ($all as $t) {
        if (!empty($t['done'])) continue;
        $remain = $t['deadline'] - time();

        if ($remain <= TASK_WARN_1H_BEFORE && empty($t['warn1h'])) {
            tg_send_kb_html(
                (int)$t['uid'],
                str_replace('{task}', $t['title'], TASK_REMINDERS['1h']),
                [[['text'=>TASK_REMINDERS['done_btn'], 'callback_data'=>"emp:task:done:".$t['id']]]]
            );
            $t['warn1h'] = true;
        }
        elseif ($remain <= TASK_WARN_5H_BEFORE && empty($t['warn5h'])) {
            tg_send_kb_html(
                (int)$t['uid'],
                str_replace('{task}', $t['title'], TASK_REMINDERS['5h']),
                [[['text'=>TASK_REMINDERS['done_btn'], 'callback_data'=>"emp:task:done:".$t['id']]]]
            );
            $t['warn5h'] = true;
        }
    }
    tasks_save($all);
}
// --- Cleanup finished tasks ---
function task_cleanup(): void {
    $all = tasks_all();
    $new = [];
    foreach ($all as $t) {
        if (!empty($t['done']) && (time() - (int)$t['done']) > 86400) {
            // پاک کردن وظایف انجام‌شده بعد از ۱ روز
            continue;
        }
        $new[] = $t;
    }
    tasks_save($new);
}

// --- Assign task to multiple users ---
function task_assign_multi(array $uids, string $title, int $deadline, string $by_role='SYSTEM'): array {
    $ids = [];
    foreach ($uids as $uid) {
        $ids[] = task_create((int)$uid, $title, $deadline, $by_role);
    }
    return $ids;
}
// --- Show tasks list for employee ---
function task_list_ui(int $chat_id, int $uid): void {
    $tasks = tasks_for_user($uid);
    if (!$tasks) {
        tg_send_html($chat_id, "🎉 هیچ وظیفه‌ای برات ثبت نشده.");
        return;
    }

    $msg = "📋 وظایف شما:\n\n";
    foreach ($tasks as $t) {
        $status = !empty($t['done']) ? "✅ انجام شد" : "⌛ در انتظار";
        $msg .= "▪️ {$t['title']} — {$status}\n";
        $msg .= "⏰ مهلت: " . dt_jalali($t['deadline']) . "\n\n";
    }
    tg_send_html($chat_id, $msg);
}
// --- Router for employee panel ---
function tasks_router(int $chat_id, int $uid, array $parts): void {
    $act = $parts[1] ?? '';
    if ($act === 'list') {
        task_list_ui($chat_id, $uid);
    }
    elseif ($act === 'done') {
        $tid = $parts[2] ?? '';
        if ($tid) task_done($uid, $tid);
    }
}

// --- پایان tasks.php ---
