<?php
declare(strict_types=1);

/**
 * Vision Media Bot — users.php (Part 1/7)
 * مدیریت کاربران و نقش‌ها
 */

const USERS_FILE = __DIR__ . '/../storage/users.json';

// --- Load / Save ---
function users_all(): array {
    return storage_load(USERS_FILE);
}
function users_save(array $all): void {
    storage_save(USERS_FILE, $all);
}
// --- Get / Save one user ---
function user_get(int $uid): ?array {
    $all = users_all();
    foreach ($all as $u) {
        if ((int)$u['id'] === $uid) return $u;
    }
    return null;
}

function user_save(array $user): void {
    $all = users_all();
    $found = false;
    foreach ($all as &$u) {
        if ((int)$u['id'] === (int)$user['id']) {
            $u = array_merge($u, $user);
            $found = true;
        }
    }
    if (!$found) $all[] = $user;
    users_save($all);
}
// --- By Role ---
function users_by_role(string $role): array {
    $all = users_all();
    return array_values(array_filter($all, fn($u) => ($u['role'] ?? '') === $role));
}

function users_has_role(int $uid, string $role): bool {
    $u = user_get($uid);
    return $u && ($u['role'] ?? '') === $role;
}

// --- Assign Role ---
function user_assign_role(int $uid, string $role): void {
    $u = user_get($uid) ?? ['id' => $uid];
    $u['role'] = $role;
    user_save($u);
}
// --- Profile helpers ---
function user_set_profile(int $uid, array $data): void {
    $u = user_get($uid) ?? ['id' => $uid];
    $u = array_merge($u, $data);
    user_save($u);
}

function user_full_name(array $u): string {
    return trim($u['full_name'] ?? '');
}

function user_role_label(array $u): string {
    $r = $u['role'] ?? '';
    $map = [
        'CEO'             => 'مدیرعامل',
        'ACCOUNTING'      => 'حسابداری',
        'DEPT_MANAGER'    => 'مدیر دپارتمان',
        'ADMIN_LEAD'      => 'ادمین ارشد',
        'ADMIN'           => 'ادمین',
        'VIDEOGRAPHER'    => 'تصویربردار',
        'BLOGGER'         => 'بلاگر',
        'EDITOR'          => 'تدوینگر',
        'SCENARIO_WRITER' => 'سناریونویس',
        'CUSTOMER'        => 'مشتری',
    ];
    return isset($map[$r]) ? $map[$r] : $r;
}

// --- Seed employees from config ---
function users_seed_from_config(): void {
    $all = users_all();
    if ($all && count($all) > 0) return; // already seeded

    foreach (EMPLOYEE_SEED as $e) {
        $all[] = [
            'id'        => $e['tg_id'],
            'full_name' => $e['full_name'],
            'role'      => $e['role'],
            'birth'     => $e['birth'] ?? '',
            'nid'       => $e['nid'] ?? '',
            'address'   => $e['address'] ?? '',
            'father'    => $e['father'] ?? '',
            'iban'      => $e['iban'] ?? '',
            'card'      => $e['card'] ?? '',
            'created'   => time(),
        ];
    }
    users_save($all);
}
// --- Search users ---
function users_search(string $q): array {
    $all = users_all();
    $q = mb_strtolower(trim($q));
    return array_values(array_filter($all, function($u) use ($q) {
        return str_contains(mb_strtolower($u['full_name'] ?? ''), $q)
            || str_contains((string)($u['id'] ?? ''), $q)
            || str_contains(mb_strtolower($u['role'] ?? ''), $q);
    }));
}

// --- Delete user ---
function user_delete(int $uid): void {
    $all = users_all();
    $all = array_values(array_filter($all, fn($u) => (int)$u['id'] !== $uid));
    users_save($all);
}
// --- Helpers for roles ---
function role_is_manager(string $role): bool {
    return in_array($role, ['CEO','DEPT_MANAGER','ACCOUNTING']);
}

function role_is_staff(string $role): bool {
    return in_array($role, ['ADMIN_LEAD','ADMIN','VIDEOGRAPHER','BLOGGER','EDITOR','SCENARIO_WRITER']);
}

function role_is_customer(string $role): bool {
    return $role === 'CUSTOMER';
}

// --- پایان users.php ---
