<?php
declare(strict_types=1);

/**
 * Vision Media Bot — utils.php (Part 1/7)
 * توابع کمکی عمومی
 */

// --- String Helpers ---
function str_has(string $haystack, string $needle): bool {
    return mb_strpos($haystack, $needle) !== false;
}
function str_eq(string $a, string $b): bool {
    return mb_strtolower(trim($a)) === mb_strtolower(trim($b));
}
function str_clean(string $txt): string {
    return trim(preg_replace('/\s+/', ' ', $txt));
}
// --- Array & JSON Helpers ---
function arr_get(array $a, string $k, $def = null) {
    return $a[$k] ?? $def;
}

function json_pp($data): string {
    return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
}

function load_json(string $file): array {
    if (!file_exists($file)) return [];
    $txt = file_get_contents($file);
    $d   = json_decode($txt, true);
    return $d ?: [];
}

function save_json(string $file, array $data): void {
    file_put_contents($file, json_pp($data));
}
// --- Time Helpers ---
function now(): int {
    return time();
}

function dt_format(int $ts): string {
    return date('Y/m/d H:i', $ts);
}

function dt_jalali(int $ts): string {
    return jdate('Y/m/d H:i', $ts);
}

function days_after(int $days): int {
    return time() + ($days * 86400);
}

function hours_after(int $h): int {
    return time() + ($h * 3600);
}
// --- Currency Helpers ---
function money(int $amt): string {
    return number_format($amt) . ' ' . CURRENCY;
}

// --- Greeting Helpers ---
function greet_now(string $name = ''): string {
    return vm_polite_header($name);
}

// --- Cleanup Chat (simulate auto delete) ---
function schedule_cleanup(int $chat_id, int $msg_id): void {
    // در تلگرام Bot API حذف پیام با deleteMessage است؛
    // اینجا فقط لاگ می‌زنیم تا کران حذف کند.
    $all = load_json(STORAGE['schedules']);
    $all[] = [
        'type'   => 'delete_msg',
        'chat'   => $chat_id,
        'msg_id' => $msg_id,
        'time'   => time() + AUTO_DELETE_INTERVAL,
    ];
    save_json(STORAGE['schedules'], $all);
}
// --- Logging Helpers ---
function log_info(string $msg): void {
    file_put_contents(LOG_DIR . '/info-' . date('Y-m-d') . '.log',
        "[" . date('H:i:s') . "] " . $msg . "\n",
        FILE_APPEND
    );
}

function log_error(string $msg): void {
    file_put_contents(LOG_DIR . '/error-' . date('Y-m-d') . '.log',
        "[" . date('H:i:s') . "] " . $msg . "\n",
        FILE_APPEND
    );
}

// --- ID Generators ---
function new_id(string $prefix): string {
    return $prefix . '_' . bin2hex(random_bytes(4)) . '_' . time();
}
// --- Role & Access Helpers ---
function is_ceo(array $u): bool {
    return ($u['role'] ?? '') === 'CEO';
}
function is_accounting(array $u): bool {
    return ($u['role'] ?? '') === 'ACCOUNTING';
}
function is_dept_manager(array $u): bool {
    return ($u['role'] ?? '') === 'DEPT_MANAGER';
}
function is_employee(array $u): bool {
    $r = $u['role'] ?? '';
    return in_array($r, ['ADMIN_LEAD','ADMIN','VIDEOGRAPHER','BLOGGER','EDITOR','SCENARIO_WRITER']);
}
function is_customer(array $u): bool {
    return ($u['role'] ?? '') === 'CUSTOMER';
}
// --- Misc ---
function arr_find(array $arr, callable $fn) {
    foreach ($arr as $item) {
        if ($fn($item)) return $item;
    }
    return null;
}

function arr_filter_map(array $arr, callable $fn): array {
    $out = [];
    foreach ($arr as $item) {
        $r = $fn($item);
        if ($r !== null) $out[] = $r;
    }
    return $out;
}

// --- پایان utils.php ---
