<?php
declare(strict_types=1);

/**
 * Vision Media Bot — handlers/ceo.php (FULL, minimal UI + impersonation)
 * پنل مدیرعامل: ورود/خروج حالت نمایشی همه نقش‌ها، گزارش سریع، لیست‌ها
 */

/* ============ Helpers for impersonation ============ */

function ceo_impersonation_active(int $uid): bool {
    $s = conv_get($uid);
    return $s && ($s['flow'] ?? '') === 'impersonate' && (int)($s['data']['cust_id'] ?? 0) > 0;
}
function ceo_impersonated_user(int $uid): ?array {
    $s = conv_get($uid);
    if (!$s || ($s['flow'] ?? '') !== 'impersonate') return null;
    $id = (int)($s['data']['cust_id'] ?? 0);
    return $id ? (user_get($id) ?: ['id'=>$id,'full_name'=>'کاربر','role'=>null]) : null;
}
function ceo_enter_impersonate(int $ceo_id, int $target_user_id): void {
    conv_start($ceo_id, 'impersonate', 0, ['cust_id'=>$target_user_id]);
}
function ceo_exit_impersonate(int $ceo_id): void {
    $s = conv_get($ceo_id);
    if ($s && ($s['flow'] ?? '') === 'impersonate') conv_clear($ceo_id);
}

/* ============ UI helpers ============ */

function role_label(string $r): string {
    switch ($r) {
        case 'ACCOUNTING': return 'حسابداری';
        case 'DEPT_MANAGER': return 'مدیر دپارتمان';
        case 'ADMIN_LEAD': return 'ادمین ارشد';
        case 'ADMIN': return 'ادمین';
        case 'VIDEOGRAPHER': return 'تصویربردار';
        case 'BLOGGER': return 'بلاگر';
        case 'EDITOR': return 'تدوینگر';
        case 'SCENARIO_WRITER': return 'سناریونویس';
        case 'CUSTOMER': return 'مشتری';
        default: return $r;
    }
}

/* ============ Main menu ============ */

function ceo_menu(int $chat_id, array $user): void {
    $rows = [];
    $banner = '';

    if (ceo_impersonation_active((int)$user['id'])) {
        $imp = ceo_impersonated_user((int)$user['id']);
        $banner = "⚠️ حالت نمایشی: <b>".htmlspecialchars($imp['full_name'] ?? $imp['id'])."</b>"
                . " (ID: ".($imp['id'] ?? '-').") — نقش: ".user_role_label($imp)."\n\n";
        $rows[] = [ kb_btn("🚪 خروج از حالت نمایشی", "ceo:as:exit") ];
    }

    // نقش‌ها (دو ستونه)
    $rows[] = [ kb_btn("🧾 حسابداری", "ceo:role:ACCOUNTING"), kb_btn("🗂️ دپارتمان", "ceo:role:DEPT_MANAGER") ];
    $rows[] = [ kb_btn("⭐ ادمین ارشد", "ceo:role:ADMIN_LEAD"), kb_btn("👨‍💻 ادمین", "ceo:role:ADMIN") ];
    $rows[] = [ kb_btn("🎥 تصویربردار", "ceo:role:VIDEOGRAPHER"), kb_btn("🧑‍🎤 بلاگر", "ceo:role:BLOGGER") ];
    $rows[] = [ kb_btn("✂️ تدوینگر", "ceo:role:EDITOR"), kb_btn("📝 سناریو", "ceo:role:SCENARIO_WRITER") ];
    $rows[] = [ kb_btn("🤝 مشتری", "ceo:role:CUSTOMER") ];
    // مدیریتی
    $rows[] = [ kb_btn("👥 پرسنل", "ceo:staff"), kb_btn("📊 گزارش", "ceo:report") ];

    tg_send_kb_html(
        $chat_id,
        vm_polite_header($user['full_name'])
        . $banner
        . "👑 <b>پنل مدیرعامل</b>\nلطفاً انتخاب کنید:",
        kb_inline($rows)
    );
}

/* ============ Router ============ */

function ceo_router(int $chat_id, int $uid, array $user, array $parts): void {
    $act = $parts[1] ?? '';

    if ($act === 'staff')     { ceo_list_staff($chat_id); return; }
    if ($act === 'report')    { ceo_quick_report($chat_id); return; }

    if ($act === 'role') {
        $role = $parts[2] ?? '';
        if (!$role) { tg_send_alert($chat_id, "نقش نامعتبر."); return; }
        ceo_list_by_role($chat_id, $role, $role === 'CUSTOMER' ? "لیست مشتریان" : "لیست نقش «".role_label($role)."»");
        return;
    }

    if ($act === 'as') {
        $sub = $parts[2] ?? '';
        if ($sub === 'exit') {
            ceo_exit_impersonate($uid);
            tg_send_success($chat_id, "از حالت نمایشی خارج شدی.");
            ceo_menu($chat_id, $user);
            return;
        }
        $target = (int)$sub;
        if ($target <= 0) { tg_send_alert($chat_id, "شناسه نامعتبر."); return; }
        $tu = user_get($target); if (!$tu) { tg_send_alert($chat_id, "کاربر پیدا نشد."); return; }
        ceo_enter_impersonate($uid, $target);
        tg_send_success($chat_id, "✅ ورود به پنل «".htmlspecialchars($tu['full_name'] ?? $target)."» (نقش: ".user_role_label($tu).")");

        // هدایت به منوی همان نقش
        $role = $tu['role'] ?? '';
        if ($role === 'CUSTOMER') { customer_menu($chat_id, $tu); return; }
        if ($role === 'ACCOUNTING') { acc_menu($chat_id); return; }
        if ($role === 'DEPT_MANAGER') { dept_menu($chat_id, $tu); return; }
        if ($role === 'ADMIN_LEAD' || $role === 'ADMIN') { admin_menu($chat_id, $tu); return; }
        if (in_array($role, ['VIDEOGRAPHER','BLOGGER','EDITOR','SCENARIO_WRITER'])) { employee_menu($chat_id, $tu); return; }

        ceo_menu($chat_id, $user);
        return;
    }

    ceo_menu($chat_id, $user);
}

/* ============ Lists ============ */

function ceo_list_by_role(int $chat_id, string $role, string $title=''): void {
    $users = users_by_role($role);
    if (!$users) { tg_send_html($chat_id, "موردی برای «".role_label($role)."» یافت نشد."); return; }

    $msg = "📋 <b>".($title ?: "لیست کاربران")."</b>\n\n";
    $rows = [];
    foreach ($users as $u) {
        $msg .= "• ".($u['full_name'] ?? $u['id'])." — ID: ".$u['id']." — نقش: ".user_role_label($u)."\n";
        $rows[] = [ kb_btn("ورود به «".($u['full_name'] ?? $u['id'])."»", "ceo:as:".$u['id']) ];
    }
    $rows[] = [ kb_btn("↩️ بازگشت", "ceo:menu") ];
    tg_send_kb_html($chat_id, $msg, kb_inline($rows));
}

function ceo_list_staff(int $chat_id): void {
    $all = users_all();
    $staff = array_values(array_filter($all, function($u){
        $r = $u['role'] ?? '';
        return $r && $r !== 'CUSTOMER';
    }));
    if (!$staff) { tg_send_html($chat_id, "هیچ پرسنلی ثبت نشده."); return; }

    $msg = "👥 <b>لیست پرسنل</b>\n\n";
    foreach ($staff as $u) {
        $msg .= "• ".($u['full_name'] ?? $u['id'])." — نقش: ".user_role_label($u)." — ID: ".$u['id']."\n";
    }
    tg_send_html($chat_id, $msg);
}

/* ============ Report ============ */

function ceo_quick_report(int $chat_id): void {
    $all  = users_all();
    $staff= array_filter($all, function($u){ $r=$u['role']??''; return $r && $r !== 'CUSTOMER'; });
    $custs= users_by_role('CUSTOMER');

    $invs = storage_load(STORAGE['invoices']);
    $pens = storage_load(STORAGE['penalties']);

    $total_invoices = 0;
    foreach ($invs as $i) $total_invoices += (int)($i['amount'] ?? 0);

    $rows = [
        [ kb_btn("🧾 حسابداری", "acc:report"), kb_btn("↩️ بازگشت", "ceo:menu") ]
    ];
    $msg = "📊 <b>گزارش سریع</b>\n\n"
         . "پرسنل: ".count($staff)." نفر\n"
         . "مشتریان: ".count($custs)." نفر\n"
         . "تعداد فاکتورها: ".count($invs)."\n"
         . "جمع مبلغ فاکتورها: ".money((int)$total_invoices)."\n"
         . "تعداد جریمه‌ها: ".count($pens)."\n";
    tg_send_kb_html($chat_id, $msg, kb_inline($rows));
}
