<?php
declare(strict_types=1);

/**
 * Vision Media Bot — handlers/department.php (FULL, minimal, integrated)
 * پنل مدیر دپارتمان:
 * - تشکیل تیم مشتری (انتخاب افراد هر نقش)
 * - ثبت آفیش و اعلان به مشتری/تصویربردار/بلاگر + تایید/رد عوامل
 * - یادآوری حضور سر ساعت آفیش + ثبت شروع/پایان با حداقل 30 دقیقه
 * - ساخت خودکار تسک سناریونویس (30 ساعت) و تسک ارسال فایل‌ها (24 ساعت) برای تصویربردار
 * - SLA: اگر تا 7 روز بعد از تخصیص مشتری، اولین آفیش ثبت نشود → هشدار به مدیر، حسابداری و مدیرعامل
 *
 * Storage keys:
 * - STORAGE['dept_customers']: [{cust, manager, assigned_at}]
 * - STORAGE['teams']: [{cust, manager, videographer, blogger, editor, scenario, admin, admin_lead}]
 * - STORAGE['shoots']: [{
 *      id,cust,manager,when_ts,when_str,address,location,status,
 *      v_id,b_id,created_at,
 *      presence_asked, start_v, end_v, start_b, end_b, end_btn_v, end_btn_b
 *   }]
 */

/* ======================= UI: منو ======================= */

function dept_menu(int $chat_id, array $user): void {
    $rows = [
        [ kb_btn("🤝 مشتریان من", "dept:custs"), kb_btn("👥 تشکیل تیم", "dept:team:new") ],
        [ kb_btn("🎬 ثبت آفیش", "dept:shoot:new"), kb_btn("🗓️ تقویم محتوا", "dept:calendar") ],
        [ kb_btn("⚠️ گزارش تأخیرها", "dept:sla") ],
    ];
    tg_send_kb_html(
        $chat_id,
        vm_polite_header($user['full_name'])."🗂️ <b>پنل مدیر دپارتمان</b>\nانتخاب کنید:",
        kb_inline($rows)
    );
}

/* ======================= Router ======================= */

function dept_router(int $chat_id, int $uid, array $user, array $parts): void {
    $act = $parts[1] ?? '';

    if ($act === 'custs')   { dept_my_customers($chat_id, $uid); return; }

    if ($act === 'team') {
        $sub = $parts[2] ?? '';
        if ($sub === 'new') { dept_team_new_start($chat_id, $uid); return; }
        // می‌توان view/edit را بعداً افزود
    }

    if ($act === 'shoot') {
        $sub = $parts[2] ?? '';
        if ($sub === 'new')     { dept_shoot_new_start($chat_id, $uid); return; }
        if ($sub === 'approve') { $id=$parts[3]??''; dept_shoot_actor_reply($chat_id,$uid,$id,true); return; }
        if ($sub === 'reject')  { $id=$parts[3]??''; dept_shoot_actor_reply($chat_id,$uid,$id,false); return; }
        if ($sub === 'here')    { $id=$parts[3]??''; dept_shoot_presence_here($chat_id,$uid,$id); return; }
        if ($sub === 'not')     { $id=$parts[3]??''; dept_shoot_presence_not($chat_id,$uid,$id); return; }
        if ($sub === 'end')     { $id=$parts[3]??''; dept_shoot_end($chat_id,$uid,$id); return; }
    }

    if ($act === 'calendar') { tg_send_html($chat_id,"🗓️ تقویم محتوا به‌زودی تکمیل می‌شود."); return; }
    if ($act === 'sla')      { dept_sla_report($chat_id, $uid); return; }

    dept_menu($chat_id, $user);
}

/* ======================= مشتریان من ======================= */

function dept_my_customers(int $chat_id, int $manager_id): void {
    $assigns = storage_load(STORAGE['dept_customers']);
    $mine = array_values(array_filter($assigns, fn($r)=>(int)$r['manager']===$manager_id));
    if (!$mine) { tg_send_html($chat_id, "فعلاً مشتریی به شما تخصیص داده نشده."); return; }

    $msg = "🤝 <b>مشتریان شما</b>\n\n";
    $rows = [];
    foreach ($mine as $r) {
        $u = user_get((int)$r['cust']) ?: ['id'=>$r['cust'],'full_name'=>'مشتری'];
        $msg .= "• ".($u['full_name']??$u['id'])." — ID: ".$u['id']."\n";
        $rows[] = [
            kb_btn("👥 تیم", "dept:team:view:".$u['id']),
            kb_btn("🎬 آفیش", "dept:shoot:new")
        ];
    }
    tg_send_kb_html($chat_id,$msg,kb_inline($rows));
}

/* ======================= تشکیل تیم مشتری ======================= */

function dept_team_new_start(int $chat_id, int $manager_id): void {
    tg_send_html($chat_id, "👤 <b>آی‌دی عددی مشتری</b> را ارسال کنید.\nمثال: <code>123456789</code>");
    conv_start($chat_id, 'dept_team_new', 0, ['manager'=>$manager_id]);
}
function dept_team_new_flow(int $chat_id, int $uid, string $text): bool {
    $s=conv_get($uid); if(!$s||($s['flow']??'')!=='dept_team_new') return false;
    $step=(int)($s['step']??0); $data=$s['data']??[];

    if ($step===0) {
        $cust=(int)preg_replace('/\D+/','',$text);
        if($cust<=0){ tg_send_alert($chat_id,"آی‌دی نامعتبر."); return true; }
        $u=user_get($cust); if(!$u){ tg_send_alert($chat_id,"مشتری یافت نشد."); return true; }
        conv_start($uid,'dept_team_new',1,['manager'=>$data['manager'],'cust'=>$cust,'team'=>[]]);
        tg_send_html($chat_id,"👌 آی‌دی <b>تصویربردار</b> را بفرست.");
        return true;
    }
    if ($step===1) {
        $v=(int)preg_replace('/\D+/','',$text);
        $team=['videographer'=>$v];
        conv_start($uid,'dept_team_new',2,['manager'=>$data['manager'],'cust'=>$data['cust'],'team'=>$team]);
        tg_send_html($chat_id,"👌 آی‌دی <b>بلاگر</b> را بفرست.");
        return true;
    }
    if ($step===2) {
        $b=(int)preg_replace('/\D+/','',$text);
        $team=$data['team']; $team['blogger']=$b;
        conv_start($uid,'dept_team_new',3,['manager'=>$data['manager'],'cust'=>$data['cust'],'team'=>$team]);
        tg_send_html($chat_id,"👌 آی‌دی <b>تدوینگر</b> را بفرست.");
        return true;
    }
    if ($step===3) {
        $e=(int)preg_replace('/\D+/','',$text);
        $team=$data['team']; $team['editor']=$e;
        conv_start($uid,'dept_team_new',4,['manager'=>$data['manager'],'cust'=>$data['cust'],'team'=>$team]);
        tg_send_html($chat_id,"👌 آی‌دی <b>سناریونویس</b> را بفرست.");
        return true;
    }
    if ($step===4) {
        $swr=(int)preg_replace('/\D+/','',$text);
        $team=$data['team']; $team['scenario']=$swr;
        conv_start($uid,'dept_team_new',5,['manager'=>$data['manager'],'cust'=>$data['cust'],'team'=>$team]);
        tg_send_html($chat_id,"👌 آی‌دی <b>ادمین</b> را بفرست.");
        return true;
    }
    if ($step===5) {
        $ad=(int)preg_replace('/\D+/','',$text);
        $team=$data['team']; $team['admin']=$ad;
        conv_start($uid,'dept_team_new',6,['manager'=>$data['manager'],'cust'=>$data['cust'],'team'=>$team]);
        tg_send_html($chat_id,"👌 آی‌دی <b>ادمین ارشد</b> را بفرست.");
        return true;
    }
    if ($step===6) {
        $adl=(int)preg_replace('/\D+/','',$text);
        $team=$data['team']; $team['admin_lead']=$adl;

        // ذخیره تیم
        $teams=storage_load(STORAGE['teams']);
        $teams = array_values(array_filter($teams, fn($t)=>(int)$t['cust']!== (int)$data['cust']));
        $teams[]=['cust'=>$data['cust'],'manager'=>$data['manager']] + $team;
        storage_save(STORAGE['teams'],$teams);

        // ثبت تخصیص (برای SLA)
        $assigns = storage_load(STORAGE['dept_customers']);
        $exists = array_values(array_filter($assigns, fn($r)=>(int)$r['cust']===(int)$data['cust']));
        if(!$exists){
            $assigns[]=['cust'=>$data['cust'],'manager'=>$data['manager'],'assigned_at'=>time()];
            storage_save(STORAGE['dept_customers'],$assigns);
        }

        conv_clear($uid);
        tg_send_success($chat_id,"✅ تیم مشتری ثبت شد. حالا از «🎬 ثبت آفیش» تاریخ اول را تنظیم کن (حداکثر ۷ روز).");
        return true;
    }
    return false;
}

/* ======================= آفیش: ساخت ======================= */

function dept_shoot_new_start(int $chat_id, int $manager_id): void {
    tg_send_html($chat_id,"👤 آی‌دی عددی مشتری برای آفیش را بفرست.");
    conv_start($chat_id,'dept_shoot_new',0,['manager'=>$manager_id]);
}
function dept_shoot_new_flow(int $chat_id, int $uid, string $text): bool {
    $s=conv_get($uid); if(!$s||($s['flow']??'')!=='dept_shoot_new') return false;
    $step=(int)($s['step']??0); $data=$s['data']??[];

    if ($step===0) {
        $cust=(int)preg_replace('/\D+/','',$text);
        if($cust<=0){ tg_send_alert($chat_id,"آی‌دی نامعتبر."); return true; }
        $team=dept_team_get($cust);
        if(!$team){ tg_send_alert($chat_id,"برای این مشتری تیم ثبت نشده."); return true; }
        conv_start($uid,'dept_shoot_new',1,['manager'=>$data['manager'],'cust'=>$cust,'team'=>$team]);
        tg_send_html($chat_id,"📅 تاریخ و ساعت حضور را به صورت «YYYY-MM-DD HH:MM» بفرست.\nمثال: <code>2025-09-10 10:30</code>");
        return true;
    }
    if ($step===1) {
        $when_str=trim($text);
        $ts=strtotime($when_str);
        if(!$ts){ tg_send_alert($chat_id,"فرمت تاریخ/ساعت صحیح نیست."); return true; }
        conv_start($uid,'dept_shoot_new',2,$data+['when_str'=>$when_str,'when_ts'=>$ts]);
        tg_send_html($chat_id,"📍 آدرس/لوکیشن حضور را بفرست.");
        return true;
    }
    if ($step===2) {
        $addr=trim($text);
        conv_start($uid,'dept_shoot_new',3,$data+['address'=>$addr]);
        tg_send_html($chat_id,"📌 لینک لوکیشن (اختیاری) را بفرست یا بنویس «-»");
        return true;
    }
    if ($step===3) {
        $loc = trim($text); if($loc==='-') $loc='';
        $d=$data; $d['location']=$loc;

        // ذخیره آفیش
        $shoots=storage_load(STORAGE['shoots']);
        $id = 'S'.time().random_int(100,999);
        $record = [
            'id'=>$id,'cust'=>$d['cust'],'manager'=>$d['manager'],
            'when_ts'=>$d['when_ts'],'when_str'=>$d['when_str'],
            'address'=>$d['address'],'location'=>$d['location'],
            'status'=>'pending','v_id'=>$d['team']['videographer']??0,'b_id'=>$d['team']['blogger']??0,
            'created_at'=>time()
        ];
        $shoots[]=$record; storage_save(STORAGE['shoots'],$shoots);

        // اطلاع‌رسانی
        $cust=user_get((int)$d['cust']) ?: ['id'=>$d['cust'],'full_name'=>'مشتری'];
        $v_id=(int)($d['team']['videographer']??0);
        $b_id=(int)($d['team']['blogger']??0);

        $info = "🎬 <b>آفیش جدید</b>\n"
              . "👤 مشتری: ".($cust['full_name']??$cust['id'])."\n"
              . "📅 زمان: {$d['when_str']}\n"
              . "📍 آدرس: {$d['address']}".($d['location']? "\n🔗 موقعیت: {$d['location']}":"")."\n"
              . "شناسه آفیش: <code>{$id}</code>";

        // مشتری
        tg_send_html((int)$cust['id'], $info."\n\n⏱️ لطفاً در زمان مقرر آماده باشید.");

        // تصویربردار
        if ($v_id>0) {
            tg_send_kb_html($v_id, $info."\n\nلطفاً تایید/رد کنید:", kb_inline([
                [ kb_btn("✅ تایید می‌کنم", "dept:shoot:approve:".$id) , kb_btn("❌ نمی‌توانم", "dept:shoot:reject:".$id) ]
            ]));
        }
        // بلاگر
        if ($b_id>0) {
            tg_send_kb_html($b_id, $info."\n\nلطفاً تایید/رد کنید:", kb_inline([
                [ kb_btn("✅ تایید می‌کنم", "dept:shoot:approve:".$id) , kb_btn("❌ نمی‌توانم", "dept:shoot:reject:".$id) ]
            ]));
        }

        // === تسک سناریونویس: 30 ساعت مهلت
        $scn_id = (int)($d['team']['scenario'] ?? 0);
        if ($scn_id > 0) {
            $title = "سناریو برای آفیش {$id}";
            $desc  = "لطفاً سناریوی این آفیش را آماده کن و تعداد «پست/ریلز»، «استوری»، «کاور» را مشخص و اعلام کن.";
            $due   = time() + (30 * 3600);
            task_create($scn_id, $title, $desc, $due, [
                'type' => 'scenario_write',
                'shoot_id' => $id,
            ]);
        }

        conv_clear($uid);
        tg_send_success($chat_id,"✅ آفیش با شناسه {$id} ثبت و اطلاع‌رسانی شد.");
        return true;
    }

    return false;
}

function dept_shoot_actor_reply(int $chat_id, int $actor_id, string $shoot_id, bool $accept): void {
    $shoots=storage_load(STORAGE['shoots']);
    foreach ($shoots as &$s) {
        if (($s['id']??'')===$shoot_id) {
            $role = ($actor_id===(int)($s['v_id']??0)) ? 'videographer' : (($actor_id===(int)($s['b_id']??0))?'blogger':'other');
            if ($role==='other'){ tg_send_alert($chat_id,"شما عضو این آفیش نیستید."); return; }
            $key = $role.'_ok';
            $s[$key] = $accept ? 1 : -1;
            storage_save(STORAGE['shoots'],$shoots);

            $manager=(int)($s['manager']??0);
            $txt = $accept ? "✅ تایید کرد" : "❌ رد کرد";
            tg_send_html($manager, "اطلاع: ".($role==='videographer'?'تصویربردار':'بلاگر')." آفیش {$shoot_id} را {$txt}.");
            tg_send_success($chat_id, $accept? "سپاس! تایید شد ✅":"ثبت شد. مدیر مطلع شد ❌");
            return;
        }
    }
    tg_send_alert($chat_id,"آفیش پیدا نشد.");
}

/* ======================= حضور و اتمام ======================= */

function dept_presence_cron(): void {
    $shoots = storage_load(STORAGE['shoots']);
    $now = time();
    $changed = false;

    foreach ($shoots as &$s) {
        // ارسال پرسش حضور در زمان آفیش (یک بار)
        if (($s['when_ts'] ?? 0) <= $now && empty($s['presence_asked'])) {
            $msg = "⏰ <b>یادآوری حضور آفیش</b>\nشناسه: <code>{$s['id']}</code>\n"
                 . "زمان: ".($s['when_str'] ?? '')."\n"
                 . "آدرس: ".($s['address'] ?? '');
            $kb = kb_inline([[ kb_btn("✅ رسیدم", "dept:shoot:here:".$s['id']),
                               kb_btn("❌ نرسیدم", "dept:shoot:not:".$s['id']) ]]);

            if (!empty($s['v_id'])) tg_send_kb_html((int)$s['v_id'], $msg, $kb);
            if (!empty($s['b_id'])) tg_send_kb_html((int)$s['b_id'], $msg, $kb);

            $s['presence_asked'] = 1;
            $changed = true;
        }

        // بعد از 30 دقیقه از شروع حضور، دکمه «اتمام آفیش» را ارسال کن (برای هر نقش یک‌بار)
        foreach (['v' => 'v_id', 'b' => 'b_id'] as $k => $rid) {
            $startKey = "start_{$k}";
            $endBtn   = "end_btn_{$k}";
            if (!empty($s[$startKey]) && empty($s[$endBtn]) && ($now - (int)$s[$startKey]) >= 1800) {
                $who = ($k==='v' ? 'تصویربردار' : 'بلاگر');
                $kb = kb_inline([[ kb_btn("🛑 اتمام آفیش", "dept:shoot:end:".$s['id']) ]]);
                $to = (int)($s[$rid] ?? 0);
                if ($to > 0) tg_send_kb_html($to, "⌛️ {$who} عزیز، می‌تونی در هر لحظه «اتمام آفیش» را بزنی.", $kb);
                $s[$endBtn] = 1;
                $changed = true;
            }
        }
    }

    if ($changed) storage_save(STORAGE['shoots'], $shoots);
}

function dept_shoot_presence_here(int $chat_id, int $actor_id, string $shoot_id): void {
    $shoots = storage_load(STORAGE['shoots']);
    foreach ($shoots as &$s) {
        if (($s['id'] ?? '') === $shoot_id) {
            $role = ($actor_id===(int)($s['v_id']??0)) ? 'v' : (($actor_id===(int)($s['b_id']??0))?'b':'');
            if (!$role) { tg_send_alert($chat_id,"شما عضو این آفیش نیستید."); return; }
            $key = "start_{$role}";
            if (!empty($s[$key])) { tg_send_info($chat_id,"حضور شما قبلاً ثبت شده."); return; }
            $s[$key] = time();
            storage_save(STORAGE['shoots'], $shoots);

            $label = ($role==='v'?'تصویربردار':'بلاگر');
            tg_send_success($chat_id, "حضور شما ثبت شد ✅ (شناسه {$shoot_id})");
            $mgr = (int)($s['manager'] ?? 0);
            if ($mgr>0) tg_send_html($mgr, "ℹ️ {$label} برای آفیش {$shoot_id} رسید.");
            return;
        }
    }
    tg_send_alert($chat_id,"آفیش پیدا نشد.");
}

function dept_shoot_presence_not(int $chat_id, int $actor_id, string $shoot_id): void {
    $shoots = storage_load(STORAGE['shoots']);
    foreach ($shoots as $s) {
        if (($s['id'] ?? '') === $shoot_id) {
            $mgr = (int)($s['manager'] ?? 0);
            tg_send_alert($chat_id, "ثبت شد. لطفاً سریعاً با مدیر هماهنگ شو.");
            if ($mgr>0) tg_send_html($mgr, "🚨 کاربر {$actor_id} برای آفیش {$shoot_id} هنوز نرسیده—تماس بگیرید.");
            return;
        }
    }
    tg_send_alert($chat_id,"آفیش پیدا نشد.");
}

function dept_shoot_end(int $chat_id, int $actor_id, string $shoot_id): void {
    $shoots = storage_load(STORAGE['shoots']);
    $changed = false;

    foreach ($shoots as &$s) {
        if (($s['id'] ?? '') !== $shoot_id) continue;

        $isV = $actor_id===(int)($s['v_id']??0);
        $isB = $actor_id===(int)($s['b_id']??0);
        if (!$isV && !$isB) { tg_send_alert($chat_id,"شما عضو این آفیش نیستید."); return; }

        $role = $isV ? 'v' : 'b';
        $startKey = "start_{$role}";
        $endKey   = "end_{$role}";
        if (empty($s[$startKey])) { tg_send_alert($chat_id,"اول «رسیدم» را بزنید."); return; }

        $now = time();
        if ($now - (int)$s[$startKey] < 1800) {
            $left = 1800 - ($now - (int)$s[$startKey]);
            tg_send_info($chat_id, "هنوز زود است. حدود ".ceil($left/60)." دقیقه دیگر دوباره تلاش کن.");
            return;
        }

        if (!empty($s[$endKey])) { tg_send_info($chat_id,"پایان آفیش قبلاً ثبت شده."); return; }

        $s[$endKey] = $now;
        storage_save(STORAGE['shoots'], $shoots);
        $changed = true;

        $label = $isV ? 'تصویربردار' : 'بلاگر';
        tg_send_success($chat_id, "پایان آفیش ثبت شد ✅");

        // اگر تصویربردار بود: تسک «ارسال فایل‌ها» ظرف 24 ساعت
        if ($isV) {
            $title = "ارسال فایل‌های آفیش {$shoot_id} به تدوینگر";
            $desc  = "لطفاً تمام فایل‌های تصویربرداری این آفیش را به تدوینگر تیم مشتری تحویل بده و این تسک را تیک بزن.";
            $due   = time() + 24*3600;
            task_create($actor_id, $title, $desc, $due, [
                'type' => 'deliver_footage',
                'shoot_id' => $shoot_id,
                'auto_penalty' => 'general_250k',
            ]);
        }

        // اطلاع به مدیر
        $mgr = (int)($s['manager'] ?? 0);
        if ($mgr>0) tg_send_html($mgr, "🛑 {$label} اعلام پایان برای آفیش {$shoot_id} ثبت کرد.");
        break;
    }

    if (!$changed) tg_send_alert($chat_id,"آفیش پیدا نشد یا تغییری ثبت نشد.");
}

/* ======================= SLA هفت‌روزه ======================= */

function dept_sla_report(int $chat_id, int $manager_id): void {
    $assigns = storage_load(STORAGE['dept_customers']);
    $shoots  = storage_load(STORAGE['shoots']);
    $mine = array_values(array_filter($assigns, fn($r)=>(int)$r['manager']===$manager_id));
    if(!$mine){ tg_send_html($chat_id,"موردی نیست."); return; }

    $msg="⚠️ <b>وضعیت ۷ روزه‌ی اولین آفیش</b>\n\n";
    foreach ($mine as $r) {
        $cust=(int)$r['cust'];
        $first = arr_find($shoots, fn($s)=>(int)$s['cust']===$cust);
        $u = user_get($cust) ?: ['id'=>$cust,'full_name'=>'مشتری'];
        if ($first) {
            $msg.="• ".($u['full_name']??$u['id']).": اولین آفیش ثبت شده — ".($first['when_str']??'')."\n";
        } else {
            $days = floor((time() - (int)$r['assigned_at'])/86400);
            $warn = $days>=7 ? "🚨 دیرکرد {$days} روز" : "⏳ گذشته: {$days} روز";
            $msg.="• ".($u['full_name']??$u['id']).": هنوز آفیشی ثبت نشده — {$warn}\n";
        }
    }
    tg_send_html($chat_id,$msg);
}

function dept_sla_cron(): void {
    $assigns = storage_load(STORAGE['dept_customers']);
    $shoots  = storage_load(STORAGE['shoots']);
    foreach ($assigns as $r) {
        $cust=(int)$r['cust']; $mgr=(int)$r['manager'];
        $first = arr_find($shoots, fn($s)=>(int)$s['cust']===$cust);
        if ($first) continue;
        if (time() - (int)$r['assigned_at'] > 7*86400) {
            $u = user_get($cust) ?: ['id'=>$cust,'full_name'=>'مشتری'];
            tg_send_html($mgr, "🚨 <b>اخطار SLA</b>\nبرای مشتری ".($u['full_name']??$cust)." هنوز اولین آفیش ثبت نشده.");
            foreach (users_by_role('ACCOUNTING') as $a) {
                tg_send_html((int)$a['id'], "🚨 اخطار SLA: مدیر {$mgr} هنوز برای مشتری {$cust} اولین آفیش را ثبت نکرده.");
            }
            tg_send_html(CEO_ID, "🚨 اخطار SLA: مدیر {$mgr} — مشتری {$cust} بدون آفیشِ اول.");
        }
    }
}

/* ======================= Helpers ======================= */

function dept_team_get(int $cust_id): ?array {
    $teams=storage_load(STORAGE['teams']);
    foreach ($teams as $t) if ((int)($t['cust']??0)===$cust_id) return $t;
    return null;
}

/* ======================= Conversation dispatcher ======================= */

function dept_conv_dispatch_text(int $chat_id, int $uid, string $text): bool {
    if (dept_team_new_flow($chat_id,$uid,$text)) return true;
    if (dept_shoot_new_flow($chat_id,$uid,$text)) return true;
    return false;
}
