<?php
declare(strict_types=1);

/**
 * Vision Media Bot — handlers/employee.php (FULL, minimal)
 * نقش‌های کارمندی: سناریونویس، تصویربردار، بلاگر، تدوینگر، ادمین‌ها
 * قابلیت‌ها:
 * - سناریونویس: ثبت سناریو (تعداد پست/استوری/کاور) برای هر آفیش
 * - تصویربردار: تایید «ارسال فایل‌ها» پس از اتمام آفیش (می‌بندد تسک 24ساعته)
 * - تدوینگر: بررسی فایل‌ها ظرف 24ساعت و تایید/نقص (اگر دیرکرد → جریمه عمومی توسط سیستم تسک)
 */

function employee_menu(int $chat_id, array $user): void {
    $rows = [
        [ kb_btn("📋 وظایف من", "emp:tasks"), kb_btn("💸 فیش حقوقی", "emp:salary") ],
        [ kb_btn("🗓️ مرخصی‌ها", "emp:leave"), kb_btn("🧾 جریمه‌ها", "emp:penalties") ],
    ];
    tg_send_kb_html(
        $chat_id,
        vm_polite_header($user['full_name'])."👤 <b>پنل کارمند</b>\nگزینه‌ای را انتخاب کن:",
        kb_inline($rows)
    );
}

/* ======================= Router ======================= */

function employee_router(int $chat_id, int $uid, array $user, array $parts): void {
    $act = $parts[1] ?? '';

    if ($act === 'tasks')     { emp_tasks_list($chat_id, $uid); return; }
    if ($act === 'salary')    { emp_salary_info($chat_id, $uid); return; }
    if ($act === 'leave')     { emp_leave_menu($chat_id, $uid); return; }
    if ($act === 'penalties') { emp_penalties_list($chat_id, $uid); return; }

    // سناریونویس
    if ($act === 'scn') {
        $sub = $parts[2] ?? '';
        if ($sub === 'submit') { $shoot_id = $parts[3] ?? ''; emp_scn_submit_start($chat_id, $uid, $shoot_id); return; }
    }

    // تصویربردار: تایید ارسال فایل‌ها
    if ($act === 'deliver') {
        $shoot_id = $parts[2] ?? '';
        if ($shoot_id) { emp_v_footage_delivered($chat_id, $uid, $shoot_id); return; }
    }

    // تدوینگر: بررسی فایل‌ها
    if ($act === 'edit') {
        $sub = $parts[2] ?? '';
        $shoot_id = $parts[3] ?? '';
        if ($sub === 'check' && $shoot_id) { emp_editor_check_start($chat_id, $uid, $shoot_id); return; }
    }

    employee_menu($chat_id, $user);
}

/* ======================= وظایف / حقوق / مرخصی / جریمه‌ها (placeholder) ======================= */

function emp_tasks_list(int $chat_id, int $uid): void {
    $tasks = tasks_by_user($uid);
    if (!$tasks) { tg_send_html($chat_id,"فعلاً وظیفه‌ای نداری. ✨"); return; }
    $msg = "📋 <b>وظایف شما</b>\n\n";
    foreach ($tasks as $t) {
        $due = dt_jalali((int)($t['due'] ?? time()));
        $msg .= "• {$t['title']} — تا: {$due}\n";
        if (!empty($t['meta']['type']) && $t['meta']['type']==='deliver_footage' && !empty($t['meta']['shoot_id'])) {
            $sid = $t['meta']['shoot_id'];
            $msg .= "  └ ". "📤 ". '<a href="tg://resolve">ارسال فایل‌ها انجام شد</a>' ."\n";
            // دکمه واقعی:
            tg_send_kb_html($chat_id, "برای آفیش {$sid} اگر ارسال کردی:", kb_inline([
                [ kb_btn("📤 تایید ارسال فایل‌ها", "emp:deliver:".$sid) ]
            ]));
        }
        if (!empty($t['meta']['type']) && $t['meta']['type']==='scenario_write' && !empty($t['meta']['shoot_id'])) {
            $sid = $t['meta']['shoot_id'];
            tg_send_kb_html($chat_id, "✍️ سناریو آفیش {$sid} را ثبت کن:", kb_inline([
                [ kb_btn("📝 ثبت سناریو این آفیش", "emp:scn:submit:".$sid) ]
            ]));
        }
        if (!empty($t['meta']['type']) && $t['meta']['type']==='check_footage' && !empty($t['meta']['shoot_id'])) {
            $sid = $t['meta']['shoot_id'];
            tg_send_kb_html($chat_id, "🔎 بررسی فایل‌های آفیش {$sid}:", kb_inline([
                [ kb_btn("✅ تایید/❌ نقص", "emp:edit:check:".$sid) ]
            ]));
        }
    }
    tg_send_html($chat_id, $msg);
}
function emp_salary_info(int $chat_id, int $uid): void {
    tg_send_html($chat_id,"💸 فیش حقوقی به‌زودی از کارتابل حسابداری نمایش داده می‌شود.");
}
function emp_leave_menu(int $chat_id, int $uid): void {
    tg_send_html($chat_id,"🗓️ درخواست مرخصی به‌زودی فعال می‌شود (ارسال برای مدیر → حسابداری → مدیرعامل).");
}
function emp_penalties_list(int $chat_id, int $uid): void {
    $pens = penalties_by_user($uid);
    if (!$pens) { tg_send_html($chat_id,"✅ هیچ جریمه‌ای نداری."); return; }
    $msg="🧾 <b>جریمه‌های شما</b>\n\n";
    foreach ($pens as $p) {
        $msg .= "• ".dt_jalali((int)$p['time'])." — مبلغ: ".money((int)$p['amount'])."\n"
              . "  دلیل: ".htmlspecialchars((string)($p['reason']??'-'))."\n";
    }
    tg_send_html($chat_id,$msg);
}

/* ======================= سناریونویس: ثبت سناریو ======================= */

function emp_scn_submit_start(int $chat_id, int $uid, string $shoot_id): void {
    if (!$shoot_id) { tg_send_alert($chat_id,"شناسه آفیش نامعتبر."); return; }
    $s = shoots_get($shoot_id);
    if (!$s) { tg_send_alert($chat_id,"آفیش پیدا نشد."); return; }
    conv_start($uid, 'emp_scn_submit', 0, ['shoot_id'=>$shoot_id]);
    tg_send_html($chat_id,
        "📝 <b>ثبت سناریو برای آفیش {$shoot_id}</b>\n"
      . "به‌ترتیب پاسخ بده:\n"
      . "1) تعداد پست/ریلز (عدد)\n"
      . "2) تعداد استوری (عدد)\n"
      . "3) تعداد کاور (عدد)"
    );
}
function emp_scn_submit_flow(int $chat_id, int $uid, string $text): bool {
    $s = conv_get($uid); if (!$s || ($s['flow'] ?? '') !== 'emp_scn_submit') return false;
    $step = (int)($s['step'] ?? 0);
    $data = $s['data'] ?? [];
    $num  = (int)preg_replace('/\D+/', '', $text);

    if ($step === 0) {
        if ($num<=0){ tg_send_alert($chat_id,"عدد نامعتبر."); return true; }
        conv_start($uid,'emp_scn_submit',1, $data+['posts'=>$num]);
        tg_send_html($chat_id,"خوبه ✅\nحالا تعداد <b>استوری</b> را بفرست (عدد).");
        return true;
    }
    if ($step === 1) {
        if ($num<0){ tg_send_alert($chat_id,"عدد نامعتبر."); return true; }
        conv_start($uid,'emp_scn_submit',2, $data+['stories'=>$num]);
        tg_send_html($chat_id,"عالی ✅\nحالا تعداد <b>کاور</b> را بفرست (عدد).");
        return true;
    }
    if ($step === 2) {
        if ($num<0){ tg_send_alert($chat_id,"عدد نامعتبر."); return true; }
        $shoot_id = (string)$data['shoot_id'];
        $plan = ['posts'=>(int)$data['posts'],'stories'=>(int)$data['stories'],'covers'=>$num];

        // ذخیره در رکورد آفیش
        $shoots = storage_load(STORAGE['shoots']);
        foreach ($shoots as &$sh) {
            if (($sh['id'] ?? '') === $shoot_id) {
                $sh['plan'] = $plan;
                storage_save(STORAGE['shoots'], $shoots);
                break;
            }
        }

        // اطلاع به مدیر
        if (!empty($sh['manager'])) {
            tg_send_html((int)$sh['manager'],
                "📝 سناریو آفیش {$shoot_id} ثبت شد:\n"
              . "پست/ریلز: {$plan['posts']} | استوری: {$plan['stories']} | کاور: {$plan['covers']}");
        }

        conv_clear($uid);
        tg_send_success($chat_id,"✅ سناریو ثبت شد. ممنون از دقتت.");
        return true;
    }
    return false;
}

/* ======================= تصویربردار: تایید ارسال فایل‌ها ======================= */

function emp_v_footage_delivered(int $chat_id, int $uid, string $shoot_id): void {
    $shoots = storage_load(STORAGE['shoots']);
    $found = false;
    foreach ($shoots as &$s) {
        if (($s['id'] ?? '') === $shoot_id) {
            if ((int)($s['v_id'] ?? 0) !== $uid) { tg_send_alert($chat_id,"این آفیش به شما تخصیص داده نشده."); return; }
            $s['footage_sent_time'] = time();
            storage_save(STORAGE['shoots'], $shoots);
            $found = true;

            // بستن تسک deliver_footage (اگر سیستم tasks تابع close را دارد)
            task_close_by_meta($uid, ['type'=>'deliver_footage','shoot_id'=>$shoot_id]);

            tg_send_success($chat_id,"✅ تایید شد: ارسال فایل‌ها برای آفیش {$shoot_id}");

            // ساخت تسک برای تدوینگر (24 ساعت بررسی)
            $editor_id = (int)($s['editor_id'] ?? 0);
            if (!$editor_id) {
                // اگر در تیم ذخیره نشده، از teams بخوان
                $team = dept_team_get((int)$s['cust']);
                $editor_id = (int)($team['editor'] ?? 0);
            }
            if ($editor_id > 0) {
                $title = "بررسی فایل‌های آفیش {$shoot_id}";
                $desc  = "فایل‌های دریافتی از تصویربردار را با سناریو مقایسه کن و نتیجه را تایید/نقص ثبت کن.";
                $due   = time() + 24*3600;
                task_create($editor_id, $title, $desc, $due, [
                    'type' => 'check_footage',
                    'shoot_id' => $shoot_id,
                    'auto_penalty' => 'general_250k',
                ]);
                tg_send_html($editor_id, "📥 فایل‌های آفیش {$shoot_id} اعلام شد که ارسال شده. لطفاً بررسی کن.");
            }
            break;
        }
    }
    if (!$found) tg_send_alert($chat_id,"آفیش پیدا نشد.");
}

/* ======================= تدوینگر: بررسی فایل‌ها ======================= */

function emp_editor_check_start(int $chat_id, int $uid, string $shoot_id): void {
    $s = shoots_get($shoot_id);
    if (!$s) { tg_send_alert($chat_id,"آفیش پیدا نشد."); return; }

    // فقط تدوینگر همون تیم حق بررسی دارد
    $team = dept_team_get((int)$s['cust']);
    $editor_id = (int)($team['editor'] ?? 0);
    if ($editor_id !== $uid) { tg_send_alert($chat_id,"این آفیش به شما تخصیص داده نشده."); return; }

    $plan = $s['plan'] ?? ['posts'=>0,'stories'=>0,'covers'=>0];
    $plan_text = "برنامه ثبت‌شده:\n"
               . "• پست/ریلز: {$plan['posts']}\n"
               . "• استوری: {$plan['stories']}\n"
               . "• کاور: {$plan['covers']}\n\n"
               . "حالا وضعیت را انتخاب کن:";

    tg_send_kb_html($chat_id, "🔎 <b>بررسی آفیش {$shoot_id}</b>\n{$plan_text}", kb_inline([
        [ kb_btn("✅ کامل دریافت شد", "emp:edit:ok:".$shoot_id), kb_btn("❌ ناقص است", "emp:edit:nok:".$shoot_id) ]
    ]));
}

function employee_router_check_buttons(string $data, int $chat_id, int $uid): bool {
    // هندل دکمه‌های OK/NOK تدوینگر
    $p = explode(':', $data);
    if (count($p) < 4) return false;
    if ($p[0] !== 'emp' || $p[1] !== 'edit') return false;

    $sub = $p[2]; $shoot_id = $p[3];
    if ($sub === 'ok') {
        emp_editor_mark_ok($chat_id, $uid, $shoot_id);
        return true;
    }
    if ($sub === 'nok') {
        emp_editor_mark_nok($chat_id, $uid, $shoot_id);
        return true;
    }
    return false;
}

function emp_editor_mark_ok(int $chat_id, int $uid, string $shoot_id): void {
    $shoots = storage_load(STORAGE['shoots']);
    foreach ($shoots as &$s) {
        if (($s['id'] ?? '') === $shoot_id) {
            $team = dept_team_get((int)$s['cust']);
            $editor_id = (int)($team['editor'] ?? 0);
            if ($editor_id !== $uid) { tg_send_alert($chat_id,"این آفیش به شما تخصیص داده نشده."); return; }

            $s['editor_check'] = ['status'=>'ok','time'=>time()];
            storage_save(STORAGE['shoots'], $shoots);

            // بستن تسک بررسی
            task_close_by_meta($uid, ['type'=>'check_footage','shoot_id'=>$shoot_id]);

            tg_send_success($chat_id,"✅ تایید شد: فایل‌ها کامل است.");
            // اطلاع به مدیر دپارتمان
            $mgr = (int)($s['manager'] ?? 0);
            if ($mgr>0) tg_send_html($mgr, "✅ تدوینگر تایید کرد: فایل‌های آفیش {$shoot_id} کامل است.");
            return;
        }
    }
    tg_send_alert($chat_id,"آفیش پیدا نشد.");
}

function emp_editor_mark_nok(int $chat_id, int $uid, string $shoot_id): void {
    $shoots = storage_load(STORAGE['shoots']);
    foreach ($shoots as &$s) {
        if (($s['id'] ?? '') === $shoot_id) {
            $team = dept_team_get((int)$s['cust']);
            $editor_id = (int)($team['editor'] ?? 0);
            if ($editor_id !== $uid) { tg_send_alert($chat_id,"این آفیش به شما تخصیص داده نشده."); return; }

            $s['editor_check'] = ['status'=>'nok','time'=>time()];
            storage_save(STORAGE['shoots'], $shoots);

            // بستن تسک بررسی
            task_close_by_meta($uid, ['type'=>'check_footage','shoot_id'=>$shoot_id]);

            tg_send_alert($chat_id,"❌ ثبت شد: ناقص است. لطفاً به مدیر اطلاع دهید.");
            $mgr = (int)($s['manager'] ?? 0);
            if ($mgr>0) tg_send_html($mgr, "❌ تدوینگر اعلام کرد: فایل‌های آفیش {$shoot_id} ناقص است. بررسی و اقدام کنید.");
            return;
        }
    }
    tg_send_alert($chat_id,"آفیش پیدا نشد.");
}

/* ======================= Conversation dispatcher ======================= */

function employee_conv_dispatch_text(int $chat_id, int $uid, string $text): bool {
    if (emp_scn_submit_flow($chat_id,$uid,$text)) return true;
    return false;
}

/* ======================= Helpers ======================= */

function shoots_get(string $id): ?array {
    $shoots = storage_load(STORAGE['shoots']);
    foreach ($shoots as $s) if (($s['id'] ?? '') === $id) return $s;
    return null;
}
