<?php
declare(strict_types=1);

/**
 * Vision Media Bot — penalties.php (Part 1/7)
 * مدیریت جریمه‌ها (عمومی و دستی)
 */

const PENALTIES_FILE = __DIR__ . '/../storage/penalties.json';

// --- Load/Save ---
function pen_all(): array {
    return storage_load(PENALTIES_FILE);
}
function pen_save(array $all): void {
    storage_save(PENALTIES_FILE, $all);
}
// --- Create penalty row ---
function pen_create(int $uid, string $name, int $amount, string $reason, bool $auto=false): string {
    $all = pen_all();
    $id  = new_id('pen');
    $row = [
        'id'     => $id,
        'uid'    => $uid,
        'name'   => $name,
        'amount' => $amount,
        'reason' => $reason,
        'auto'   => $auto,
        'time'   => time(),
        'status' => 'approved', // جریمه پس از تایید نهایی
    ];
    $all[] = $row;
    pen_save($all);
    return $id;
}

// --- List by user ---
function pen_by_user(int $uid): array {
    return array_values(array_filter(pen_all(), fn($p) => (int)$p['uid'] === $uid));
}
// --- Format message ---
function pen_format_msg(array $p): string {
    return str_replace(
        ['{date}','{name}','{reason}','{amount}'],
        [jdate('Y/m/d H:i', $p['time'] ?? time()), $p['name'] ?? '', $p['reason'] ?? '', number_format((int)($p['amount'] ?? 0))],
        MESSAGES['penalty_format']
    );
}

// --- Notify accounting & CEO ---
function pen_notify_all(string $msg): void {
    tg_send_html(CEO_ID, "🔔 {$msg}");
    $accs = users_by_role('ACCOUNTING');
    foreach ($accs as $a) tg_send_html((int)$a['id'], "🔔 {$msg}");
}
// --- Public (auto) penalty helper ---
function pen_auto_public(int $uid, string $reason): string {
    $u = user_get($uid);
    if (!$u) return '';
    $id = pen_create($uid, $u['full_name'] ?? '', PENALTY_PUBLIC_DEFAULT, $reason, true);
    $msg = pen_format_msg(['time'=>time(),'name'=>$u['full_name']??'','reason'=>$reason,'amount'=>PENALTY_PUBLIC_DEFAULT]);
    tg_send_html($uid, $msg);
    pen_notify_all("جریمه خودکار برای {$u['full_name']} ثبت شد.");
    return $id;
}
// --- Manual penalty request by Dept Manager ---
function pen_request_manual(int $target_uid, int $by_manager, int $amount, string $reason, string $photo_id=''): string {
    $u = user_get($target_uid);
    if (!$u) return '';

    // تولید درخواست تایید  (ابتدا حسابداری، سپس مدیرعامل)
    $appr_id = approval_request_penalty($target_uid, $by_manager, $amount, $reason);
    $note = "درخواست جریمه دستی برای «{$u['full_name']}» به مبلغ " . money($amount) . " — علت: {$reason}";
    approval_add_note($appr_id, $by_manager, $note);

    // اطلاع‌رسانی به حسابداری
    $accs = users_by_role('ACCOUNTING');
    foreach ($accs as $a) {
        tg_send_html((int)$a['id'],
            "📥 درخواست جریمه جدید\n{$note}\n\n"
            . ($photo_id ? "🖼️ مستند تخلف پیوست شده است." : "")
        );
    }
    tg_send_html(CEO_ID, "📥 درخواست جریمه جدید: {$note}");
    return $appr_id;
}
// --- Finalize after approvals ---
function pen_finalize_from_approval(string $appr_id, int $approved_by): ?string {
    $a = approval_get($appr_id);
    if (!$a || ($a['type'] ?? '') !== 'penalty') return null;

    $uid    = (int)($a['target'] ?? 0);
    $amount = (int)($a['data']['amount'] ?? PENALTY_PUBLIC_DEFAULT);
    $reason = (string)($a['data']['reason'] ?? 'بدون توضیح');
    $u      = user_get($uid);
    if (!$u) return null;

    // ثبت جریمه قطعی
    $id = pen_create($uid, $u['full_name'] ?? '', $amount, $reason, false);

    // اطلاع‌رسانی
    $msg = pen_format_msg(['time'=>time(),'name'=>$u['full_name']??'','reason'=>$reason,'amount'=>$amount]);
    tg_send_html($uid, $msg);
    pen_notify_all("جریمه قطعی برای {$u['full_name']} تایید شد.");

    // TODO: کسر از حقوق (پس از ساخت ماژول حقوق)
    return $id;
}
// --- Auto penalties for missed tasks / deadlines ---
function pen_auto_for_task_deadline(int $uid, string $task_title): void {
    pen_auto_public($uid, "عدم انجام وظیفه: {$task_title}");
}

function pen_auto_for_uploader_delay(int $uid): void {
    pen_auto_public($uid, "تاخیر در ارسال فایل‌های آفیش");
}

function pen_auto_for_editor_review_delay(int $uid): void {
    pen_auto_public($uid, "تاخیر در بررسی فایل‌های دریافتی");
}

// --- پایان penalties.php ---
