<?php
declare(strict_types=1);

/**
 * Vision Media Bot — telegram.php (Part 1/7)
 * توابع ارتباط با Telegram Bot API
 */

// --- Core Request ---
function tg_api(string $method, array $params = []): array {
    $url = TG_API . $method;
    $ch  = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
    $res = curl_exec($ch);
    curl_close($ch);
    $data = json_decode((string)$res, true);
    return $data ?: ['ok'=>false,'result'=>null];
}
// --- Basic Senders ---
function tg_send_message(int $chat_id, string $text, array $opts = []): array {
    $p = array_merge([
        'chat_id' => $chat_id,
        'text'    => $text,
    ], $opts);
    return tg_api('sendMessage', $p);
}

function tg_send_html(int $chat_id, string $html, array $opts = []): array {
    $p = array_merge([
        'chat_id'    => $chat_id,
        'text'       => $html,
        'parse_mode' => 'HTML',
    ], $opts);
    return tg_api('sendMessage', $p);
}

function tg_send_kb_html(int $chat_id, string $html, array $kb): array {
    return tg_send_html($chat_id, $html, ['reply_markup' => json_encode($kb)]);
}
// --- Edit / Delete ---
function tg_edit_html(int $chat_id, int $msg_id, string $html, array $kb = null): array {
    $p = [
        'chat_id'    => $chat_id,
        'message_id' => $msg_id,
        'text'       => $html,
        'parse_mode' => 'HTML',
    ];
    if ($kb) $p['reply_markup'] = json_encode($kb);
    return tg_api('editMessageText', $p);
}

function tg_delete_msg(int $chat_id, int $msg_id): array {
    return tg_api('deleteMessage', [
        'chat_id'    => $chat_id,
        'message_id' => $msg_id,
    ]);
}
// --- Media Senders ---
function tg_send_photo(int $chat_id, string $file, string $caption = '', array $kb = null): array {
    $p = [
        'chat_id' => $chat_id,
        'caption' => $caption,
        'parse_mode' => 'HTML',
    ];
    if ($kb) $p['reply_markup'] = json_encode($kb);

    if (file_exists($file)) {
        $p['photo'] = new CURLFile($file);
    } else {
        $p['photo'] = $file; // URL or file_id
    }
    return tg_api('sendPhoto', $p);
}

function tg_send_doc(int $chat_id, string $file, string $caption = ''): array {
    $p = [
        'chat_id' => $chat_id,
        'caption' => $caption,
        'parse_mode' => 'HTML',
    ];
    if (file_exists($file)) {
        $p['document'] = new CURLFile($file);
    } else {
        $p['document'] = $file;
    }
    return tg_api('sendDocument', $p);
}
// --- Keyboards ---
function tg_reply_kb(int $chat_id, string $html, array $rows, bool $resize=true, bool $once=false): array {
    $kb = [
        'keyboard' => $rows,
        'resize_keyboard' => $resize,
        'one_time_keyboard' => $once,
    ];
    return tg_send_html($chat_id, $html, ['reply_markup' => json_encode($kb)]);
}

function tg_remove_kb(int $chat_id, string $html=''): array {
    $p = [
        'chat_id' => $chat_id,
        'text'    => $html ?: '⌨️',
        'reply_markup' => json_encode(['remove_keyboard'=>true]),
    ];
    return tg_api('sendMessage', $p);
}
// --- Chat Actions ---
function tg_action(int $chat_id, string $action='typing'): array {
    return tg_api('sendChatAction', [
        'chat_id' => $chat_id,
        'action'  => $action,
    ]);
}

// --- Utility senders ---
function tg_send_alert(int $cid, string $msg): void {
    tg_send_html($cid, "⚠️ {$msg}");
}

function tg_send_success(int $cid, string $msg): void {
    tg_send_html($cid, "✅ {$msg}");
}

function tg_send_info(int $cid, string $msg): void {
    tg_send_html($cid, "ℹ️ {$msg}");
}
// --- Answer callback (popup) ---
function tg_answer_cb(string $cbid, string $text='', bool $alert=false): array {
    $p = ['callback_query_id' => $cbid];
    if ($text)  $p['text'] = $text;
    if ($alert) $p['show_alert'] = true;
    return tg_api('answerCallbackQuery', $p);
}

// --- GetMe (health check) ---
function tg_getme(): array {
    return tg_api('getMe');
}

// --- پایان telegram.php ---
